/*
 * Copyright (C) 2013 Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package turtleresponse;

import java.awt.Desktop;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import static java.lang.Thread.sleep;
import java.math.BigInteger;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.Preferences;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.imageio.ImageIO;
import javax.swing.ButtonModel;
import javax.swing.JOptionPane;
import javax.swing.WindowConstants;
import sun.misc.HexDumpEncoder;

/**
 *
 * @author Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>
 */
public class TResponseFrame extends javax.swing.JFrame {

    private static final int MAXCHARS = 140;
    private static final int PORT_BROADCAST = 3100;
    private static final int PORT_SUBMISSION = 3100;
    private static final int BROADCAST_MESSAGE_LENGTH = 1024;
    private static final int RSA_KEY_LENGTH = 512;
    private static final int LISTENER_MAX_RETRY_COUNT = 20 ;
    private static final int LISTENER_TIMEOUT_MS = 1000 ;
    private final Object SERVER_ADDRESS_LOCK ;
    private InetAddress serverAddress;
    private DatagramSocket listenerSocket = null;
    private boolean terminatingListener = false;
    ListenerThread listener;
    private int channelNumber;
    public String passcodeForFeedback;
    public String nicknameForFeedback;
    private Preferences prefs;
    public RSAPublicKey publicKey;

    /**
     * Creates new form TEFeedbackDialog
     */
    public TResponseFrame() {
        initComponents();
        setResizable(false);
        
        submitButton.setEnabled(false) ;

        SERVER_ADDRESS_LOCK = new Object() ;
        
        prefs = Preferences.userNodeForPackage(this.getClass());
        restorePreferences();

        Image img;
        try {
            img = ImageIO.read(getClass().getResource("images/tfdbk-icon.png"));
            setIconImage(img);
        } catch (IOException ex) {
            System.err.println("icon not found");
        }


        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                quitButtonActionPerformed(null);
            }
        });

        serverAddress = null ;
        publicKey = null;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        numberButtonGroup = new javax.swing.ButtonGroup();
        channelButtonGroup = new javax.swing.ButtonGroup();
        submitButton = new javax.swing.JButton();
        quitButton = new javax.swing.JButton();
        nicknameTextField = new javax.swing.JTextField();
        nicknameLabel = new javax.swing.JLabel();
        messageLabel = new javax.swing.JLabel();
        jPanel1 = new javax.swing.JPanel();
        jRadioButton1 = new javax.swing.JRadioButton();
        jRadioButton2 = new javax.swing.JRadioButton();
        jRadioButton3 = new javax.swing.JRadioButton();
        jRadioButton4 = new javax.swing.JRadioButton();
        jRadioButton5 = new javax.swing.JRadioButton();
        jRadioButton6 = new javax.swing.JRadioButton();
        jRadioButton7 = new javax.swing.JRadioButton();
        jRadioButton8 = new javax.swing.JRadioButton();
        jRadioButton9 = new javax.swing.JRadioButton();
        jRadioButton10 = new javax.swing.JRadioButton();
        choiceLabel = new javax.swing.JLabel();
        passcodeLabel = new javax.swing.JLabel();
        messageTextArea = new javax.swing.JTextArea();
        passcodeTextField = new javax.swing.JTextField();
        channelLabel = new javax.swing.JLabel();
        jPanel2 = new javax.swing.JPanel();
        channelCRadioButton = new javax.swing.JRadioButton();
        channelBRadioButton = new javax.swing.JRadioButton();
        channelARadioButton = new javax.swing.JRadioButton();
        commentsTextArea = new javax.swing.JTextArea();
        jLabel1 = new javax.swing.JLabel();
        webButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setTitle("TurtleResponder");

        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("turtleresponse/TRBandle"); // NOI18N
        submitButton.setText(bundle.getString("SUBMIT")); // NOI18N
        submitButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                submitButtonActionPerformed(evt);
            }
        });

        quitButton.setText(bundle.getString("QUIT")); // NOI18N
        quitButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                quitButtonActionPerformed(evt);
            }
        });

        nicknameTextField.setText("nickname");

        nicknameLabel.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
        nicknameLabel.setText(bundle.getString("NICKNAME")); // NOI18N

        messageLabel.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
        messageLabel.setText(bundle.getString("MESSAGE")); // NOI18N

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder(""));

        numberButtonGroup.add(jRadioButton1);
        jRadioButton1.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton1.setMnemonic('1');
        jRadioButton1.setText("1/YES");

        numberButtonGroup.add(jRadioButton2);
        jRadioButton2.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton2.setMnemonic('2');
        jRadioButton2.setText("2/NO");

        numberButtonGroup.add(jRadioButton3);
        jRadioButton3.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton3.setMnemonic('3');
        jRadioButton3.setText("3");

        numberButtonGroup.add(jRadioButton4);
        jRadioButton4.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton4.setMnemonic('4');
        jRadioButton4.setText("4");

        numberButtonGroup.add(jRadioButton5);
        jRadioButton5.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton5.setMnemonic('5');
        jRadioButton5.setText("5");

        numberButtonGroup.add(jRadioButton6);
        jRadioButton6.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton6.setMnemonic('6');
        jRadioButton6.setText("6");

        numberButtonGroup.add(jRadioButton7);
        jRadioButton7.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton7.setMnemonic('7');
        jRadioButton7.setText("7");

        numberButtonGroup.add(jRadioButton8);
        jRadioButton8.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton8.setMnemonic('8');
        jRadioButton8.setText("8");

        numberButtonGroup.add(jRadioButton9);
        jRadioButton9.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton9.setMnemonic('9');
        jRadioButton9.setText("9");

        numberButtonGroup.add(jRadioButton10);
        jRadioButton10.setFont(new java.awt.Font("Lucida Grande", 1, 16)); // NOI18N
        jRadioButton10.setMnemonic('0');
        jRadioButton10.setSelected(true);
        jRadioButton10.setText("0/?");

        org.jdesktop.layout.GroupLayout jPanel1Layout = new org.jdesktop.layout.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel1Layout.createSequentialGroup()
                .add(23, 23, 23)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(jRadioButton1)
                    .add(jRadioButton6))
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(jPanel1Layout.createSequentialGroup()
                        .add(12, 12, 12)
                        .add(jRadioButton7))
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel1Layout.createSequentialGroup()
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jRadioButton2)))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(jPanel1Layout.createSequentialGroup()
                        .add(jRadioButton8)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jRadioButton9))
                    .add(jPanel1Layout.createSequentialGroup()
                        .add(jRadioButton3)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jRadioButton4)))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(jRadioButton5)
                    .add(jRadioButton10))
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        jPanel1Layout.linkSize(new java.awt.Component[] {jRadioButton1, jRadioButton10, jRadioButton2, jRadioButton3, jRadioButton4, jRadioButton5, jRadioButton6, jRadioButton7, jRadioButton8, jRadioButton9}, org.jdesktop.layout.GroupLayout.HORIZONTAL);

        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(jRadioButton1)
                    .add(jRadioButton2)
                    .add(jRadioButton3)
                    .add(jRadioButton4)
                    .add(jRadioButton5))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.UNRELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(jRadioButton6)
                    .add(jRadioButton7)
                    .add(jRadioButton8)
                    .add(jRadioButton9)
                    .add(jRadioButton10))
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        choiceLabel.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
        choiceLabel.setText(bundle.getString("CHOICE")); // NOI18N

        passcodeLabel.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
        passcodeLabel.setText(bundle.getString("PASSCODE")); // NOI18N

        messageTextArea.setColumns(20);
        messageTextArea.setLineWrap(true);
        messageTextArea.setRows(5);
        messageTextArea.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));

        passcodeTextField.setText("0000");

        channelLabel.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
        channelLabel.setText(bundle.getString("CHANNEL")); // NOI18N

        jPanel2.setBorder(javax.swing.BorderFactory.createEtchedBorder());

        channelButtonGroup.add(channelCRadioButton);
        channelCRadioButton.setText("C");
        channelCRadioButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                channelCRadioButtonActionPerformed(evt);
            }
        });

        channelButtonGroup.add(channelBRadioButton);
        channelBRadioButton.setText("B");
        channelBRadioButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                channelBRadioButtonActionPerformed(evt);
            }
        });

        channelButtonGroup.add(channelARadioButton);
        channelARadioButton.setSelected(true);
        channelARadioButton.setText("A");
        channelARadioButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                channelARadioButtonActionPerformed(evt);
            }
        });

        org.jdesktop.layout.GroupLayout jPanel2Layout = new org.jdesktop.layout.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .add(channelARadioButton)
                .add(18, 18, 18)
                .add(channelBRadioButton)
                .add(18, 18, 18)
                .add(channelCRadioButton)
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        jPanel2Layout.linkSize(new java.awt.Component[] {channelARadioButton, channelBRadioButton, channelCRadioButton}, org.jdesktop.layout.GroupLayout.HORIZONTAL);

        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel2Layout.createSequentialGroup()
                .addContainerGap(11, Short.MAX_VALUE)
                .add(jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(channelCRadioButton)
                    .add(channelBRadioButton)
                    .add(channelARadioButton))
                .addContainerGap())
        );

        commentsTextArea.setEditable(false);
        commentsTextArea.setBackground(javax.swing.UIManager.getDefaults().getColor("Panel.background"));
        commentsTextArea.setColumns(20);
        commentsTextArea.setLineWrap(true);
        commentsTextArea.setRows(5);
        commentsTextArea.setBorder(javax.swing.BorderFactory.createEtchedBorder());

        jLabel1.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
        jLabel1.setText(bundle.getString("INSTRUCTION")); // NOI18N

        webButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/turtleresponse/images/globe.png"))); // NOI18N
        webButton.setBorderPainted(false);
        webButton.setContentAreaFilled(false);
        webButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                webButtonActionPerformed(evt);
            }
        });

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .addContainerGap()
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(channelLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, choiceLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, jLabel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .add(layout.createSequentialGroup()
                        .add(nicknameLabel, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 98, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                        .add(0, 0, Short.MAX_VALUE))
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                        .add(0, 0, Short.MAX_VALUE)
                        .add(messageLabel, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 98, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING, false)
                    .add(layout.createSequentialGroup()
                        .add(webButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 56, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .add(quitButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 102, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                        .add(18, 18, 18)
                        .add(submitButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 109, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                    .add(layout.createSequentialGroup()
                        .add(2, 2, 2)
                        .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING, false)
                            .add(layout.createSequentialGroup()
                                .add(jPanel2, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .add(passcodeLabel, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 80, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                                .add(passcodeTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 119, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                            .add(org.jdesktop.layout.GroupLayout.LEADING, commentsTextArea)
                            .add(org.jdesktop.layout.GroupLayout.LEADING, messageTextArea)
                            .add(org.jdesktop.layout.GroupLayout.LEADING, nicknameTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 142, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                            .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))))
                .add(0, 11, Short.MAX_VALUE))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(layout.createSequentialGroup()
                        .add(16, 16, 16)
                        .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                            .add(passcodeTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                            .add(passcodeLabel)))
                    .add(layout.createSequentialGroup()
                        .add(22, 22, 22)
                        .add(channelLabel))
                    .add(layout.createSequentialGroup()
                        .addContainerGap()
                        .add(jPanel2, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)))
                .add(18, 18, 18)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(commentsTextArea, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 55, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(jLabel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 24, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 17, Short.MAX_VALUE)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(nicknameTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(nicknameLabel))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(messageTextArea, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 66, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(messageLabel))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.UNRELATED)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(jPanel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(choiceLabel))
                .add(18, 18, 18)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING, false)
                    .add(submitButton)
                    .add(quitButton)
                    .add(webButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 0, Short.MAX_VALUE))
                .addContainerGap(17, Short.MAX_VALUE))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void submitButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_submitButtonActionPerformed
        saveNicknamePasscodeAndChannel(
                nicknameTextField.getText().trim(),
                passcodeTextField.getText().trim(),
                channelNumber);
        submitFeedback();
    }//GEN-LAST:event_submitButtonActionPerformed

    private void quitButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_quitButtonActionPerformed
        terminatingListener=true ;
        saveNicknamePasscodeAndChannel(
                nicknameTextField.getText().trim(),
                passcodeTextField.getText().trim(),
                channelNumber);
        System.exit(0);
    }//GEN-LAST:event_quitButtonActionPerformed

    private void channelARadioButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_channelARadioButtonActionPerformed
        channelNumber = 0;
        restartListener() ;
    }//GEN-LAST:event_channelARadioButtonActionPerformed

    private void channelBRadioButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_channelBRadioButtonActionPerformed
        channelNumber = 1;
        restartListener() ;
    }//GEN-LAST:event_channelBRadioButtonActionPerformed

    private void channelCRadioButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_channelCRadioButtonActionPerformed
        channelNumber = 2;
        restartListener() ;
    }//GEN-LAST:event_channelCRadioButtonActionPerformed

    private void webButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_webButtonActionPerformed
        if (serverAddress == null) {
            JOptionPane.showMessageDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleresponse/TRBandle").getString("CANNOT FIND SERVER"),
                    "WARNING",
                    JOptionPane.WARNING_MESSAGE);
            return;
        } else {
            String url = "http://" + serverAddress.getHostAddress() + ":3080/index.html";
            if (Desktop.isDesktopSupported()) {
                try {
                    Desktop.getDesktop().browse(new URI(url));
                } catch (IOException e) {
                    ;
                } catch (URISyntaxException ex) {
                    ;
                }
            }
        }
    }//GEN-LAST:event_webButtonActionPerformed

    public void startListener() {
        if (listener != null || terminatingListener) return;
        
        terminatingListener = false;
        listener = new ListenerThread();
        listener.start();
        // wait a short time until listener is up
        try {
            sleep(50);
        } catch (InterruptedException ex) {
            ;
        }
    }

    private void restartListener() {        
        submitButton.setEnabled(false);
        forceToTerminateListener();  
        if (openListenerSocket()) {
            startListener();
            submitButton.setEnabled(true);
        }
    }

    private void forceToTerminateListener() {
        if (listener != null && listener.isAlive()) {
            terminatingListener = true;
            while (terminatingListener) {
                try {
                    sleep(500);
                } catch (InterruptedException ex) {
                    ;
                }
            }
        }
    }
    
    public boolean openListenerSocket() {
        DatagramSocket socket;
        try {
            socket = new DatagramSocket(PORT_BROADCAST + channelNumber);
        } catch (SocketException ex) {
            JOptionPane.showMessageDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleresponse/TRBandle").getString("PORT IS ALREADY IN USE"),
                    "WARNING",
                    JOptionPane.WARNING_MESSAGE);
            submitButton.setEnabled(false);
            return false;
        }
        listenerSocket = socket;
        return true;
    }

    public void restorePreferences() {
        passcodeForFeedback = prefs.get("PASSCODE", "");
        passcodeTextField.setText(passcodeForFeedback.trim());

        nicknameForFeedback = prefs.get("NICKNAME", "UNNAMED");
        nicknameTextField.setText(nicknameForFeedback.trim());

        channelNumber = Integer.valueOf(prefs.get("CHANNEL", "0"));
        switch (channelNumber) {
            case 0:
                channelARadioButton.setSelected(true);
                break;
            case 1:
                channelBRadioButton.setSelected(true);
                break;
            case 2:
                channelCRadioButton.setSelected(true);
                break;
            default:
                break;
        }
        
        int x = prefs.getInt("WIN_LOC_X", 10);
        int y = prefs.getInt("WIN_LOC_Y", 10);
        this.setLocation(x,y) ;
    }

    public void saveNicknamePasscodeAndChannel(String nickname, String passcode, int channel) {
        prefs.put("NICKNAME", nickname);
        prefs.put("PASSCODE", passcode);
        prefs.put("CHANNEL", Integer.toString(channel));
        
        Rectangle fbound = this.getBounds();
        prefs.putInt("WIN_LOC_X", fbound.x);
        prefs.putInt("WIN_LOC_Y", fbound.y);
    }

    public void setNicknamePasscodeAndChannel(String nickname, String passcode, int channel) {
        nicknameTextField.setText(nickname);
        passcodeTextField.setText(passcode);
        channelNumber = channel;
        switch (channel) {
            case 0:
                channelARadioButton.setSelected(true);
                break;
            case 1:
                channelBRadioButton.setSelected(true);
                break;
            case 2:
                channelCRadioButton.setSelected(true);
                break;
            default:
                break;
        }
    }

    public String selectedNumber() {
        ButtonModel model = numberButtonGroup.getSelection();
        return String.valueOf(model.getMnemonic() - '0');
    }

    public boolean submitFeedback() {

        BufferedReader reader;
        OutputStreamWriter writer;
        StringBuilder server_msg = new StringBuilder("");
        boolean gotResponse = false;

        if (serverAddress == null) {
            JOptionPane.showMessageDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleresponse/TRBandle").getString("CANNOT FIND SERVER"),
                    "WARNING",
                    JOptionPane.WARNING_MESSAGE);
            return false;
        }

        try {
            Socket socket;
            synchronized (SERVER_ADDRESS_LOCK) {
                socket = new Socket(serverAddress, PORT_SUBMISSION + channelNumber);
                writer = new OutputStreamWriter(socket.getOutputStream(), "utf-8");
                reader = new BufferedReader(
                        new InputStreamReader(socket.getInputStream(), "utf-8"));
            }
            String passcode = passcodeTextField.getText().trim();
            if (passcode.length() > 8) {
                passcode = passcode.substring(0, 8);
            }
            writer.write("passcode=" + encryptShortMessage(passcode) + "\r\n");

            String nickname = nicknameTextField.getText().trim();
            if (nickname.length() > 16) {
                nickname = nickname.substring(0, 16);
            }

            writer.write("nickname=" + nickname.replaceAll("[\t\r\n\"]", "") + "\r\n");

            writer.write("userid=" + encryptShortMessage(System.getProperty("user.name")) + "\r\n");

            String localHostName = "unknown";
            try {
                java.net.InetAddress localMachine = java.net.InetAddress.getLocalHost();
                localHostName = localMachine.getHostName();
            } catch (UnknownHostException ex) {;
            }
            writer.write("hostname=" + encryptShortMessage(localHostName) + "\r\n");

            writer.write("choice=" + selectedNumber() + "\r\n");

            String message = messageTextArea.getText().trim();
            message = message.replaceAll("\t", " ");
            message = message.replaceAll("[\r\n]", "<br>");
            // message = message.replaceAll("[\r\n]", "<br>");
            if (message.length() > MAXCHARS) {
                message = message.substring(0, MAXCHARS - 1);
            }
            writer.write("message=" + message + "\r\n\r\n");

            writer.flush();
            socket.shutdownOutput();

            String line;
            while ((line = reader.readLine()) != null) {
                // System.out.println(line);
                server_msg.append(line);
                gotResponse = true;
            }
        } catch (IOException ex) {
            // Logger.getLogger(TEFeedbackDialog.class.getName()).log(Level.SEVERE, null, ex);
            JOptionPane.showMessageDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleresponse/TRBandle").getString("COMMUNICATION ERROR"),
                    java.util.ResourceBundle.getBundle("turtleresponse/TRBandle").getString("WARNING"),
                    JOptionPane.WARNING_MESSAGE);
            serverAddress = null;
            return false;
        }

        if (!gotResponse) {
            server_msg.append("COMMUNICATION ERROR");
        }

        JOptionPane.showMessageDialog(
                this,
                server_msg.toString(),
                java.util.ResourceBundle.getBundle("turtleresponse/TRBandle").getString("INFORMATION"),
                JOptionPane.INFORMATION_MESSAGE);

        return true;
    }

    public void regeneratePublicKey(byte[] byteMod, byte[] byteExp) {
        BigInteger modules = new BigInteger(byteMod);
        BigInteger publicExponent = new BigInteger(byteExp);
        RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(modules, publicExponent);
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            publicKey = (RSAPublicKey) keyFactory.generatePublic(publicKeySpec);
        } catch (NoSuchAlgorithmException ex) {
            publicKey = null;
        } catch (InvalidKeySpecException ex) {
            publicKey = null;
        }
    }

    public String byte2hex(byte[] bytes) {
        if (bytes == null) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes) {
            sb.append(Integer.toHexString((b & 0xF0) >> 4));
            sb.append(Integer.toHexString(b & 0xF));
        }
        return sb.toString();
    }

    public String encryptShortMessage(String str) {
        if (publicKey == null) {
            return str;
        }
        byte[] data = str.getBytes();
        while (data.length > RSA_KEY_LENGTH / 8 - 12) {
            str = str.substring(0, str.length() / 2); // make it half
            data = str.getBytes();
        }
        byte[] encrypted = null;
        try {
            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            encrypted = cipher.doFinal(data);
        } catch (NoSuchAlgorithmException e) {
            ;
        } catch (NoSuchPaddingException e) {
            ;
        } catch (BadPaddingException e) {
            ;
        } catch (InvalidKeyException ex) {
            ;
        } catch (IllegalBlockSizeException ex) {
            ;
        }
        return byte2hex(encrypted);
    }

    private static void hexDump(byte[] dump) {
        HexDumpEncoder hexDump = new HexDumpEncoder();
        System.out.println(hexDump.encode(dump));
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JRadioButton channelARadioButton;
    private javax.swing.JRadioButton channelBRadioButton;
    private javax.swing.ButtonGroup channelButtonGroup;
    private javax.swing.JRadioButton channelCRadioButton;
    private javax.swing.JLabel channelLabel;
    private javax.swing.JLabel choiceLabel;
    private javax.swing.JTextArea commentsTextArea;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JRadioButton jRadioButton1;
    private javax.swing.JRadioButton jRadioButton10;
    private javax.swing.JRadioButton jRadioButton2;
    private javax.swing.JRadioButton jRadioButton3;
    private javax.swing.JRadioButton jRadioButton4;
    private javax.swing.JRadioButton jRadioButton5;
    private javax.swing.JRadioButton jRadioButton6;
    private javax.swing.JRadioButton jRadioButton7;
    private javax.swing.JRadioButton jRadioButton8;
    private javax.swing.JRadioButton jRadioButton9;
    private javax.swing.JLabel messageLabel;
    private javax.swing.JTextArea messageTextArea;
    private javax.swing.JLabel nicknameLabel;
    private javax.swing.JTextField nicknameTextField;
    private javax.swing.ButtonGroup numberButtonGroup;
    private javax.swing.JLabel passcodeLabel;
    private javax.swing.JTextField passcodeTextField;
    private javax.swing.JButton quitButton;
    private javax.swing.JButton submitButton;
    private javax.swing.JButton webButton;
    // End of variables declaration//GEN-END:variables

    class ListenerThread extends Thread {

        public ListenerThread() {
        }

        @Override
        public void run() {
            byte[] receiveData = new byte[BROADCAST_MESSAGE_LENGTH];
            int retry_counter=0 ;
            for (;;) {
                             
                if (terminatingListener) { // going to die
                    listenerSocket.close();
                    listener=null ;
                    terminatingListener=false ;
                    return ;
                }
                
                DatagramPacket receivePacket =
                        new DatagramPacket(receiveData, receiveData.length);

                try {
                    listenerSocket.setSoTimeout(LISTENER_TIMEOUT_MS);
                    listenerSocket.receive(receivePacket);
                } catch (UnsupportedEncodingException ex) {
                    continue;
                } catch (IOException ex) {
                     retry_counter++ ;
                    if (serverAddress != null && retry_counter < LISTENER_MAX_RETRY_COUNT) {
                        continue ;
                    } else {
                        if (serverAddress != null) {
                            synchronized (SERVER_ADDRESS_LOCK) {
                                serverAddress = null;
                            }
                        }
                        publicKey = null;
                        submitButton.setEnabled(false) ;
                        continue ;
                    } 
                }

                retry_counter=0 ;
                
                synchronized (SERVER_ADDRESS_LOCK) {
                    if (serverAddress == null) {
                        serverAddress = receivePacket.getAddress();
                    } else {
                        InetAddress addr = receivePacket.getAddress();
                        if (!addr.equals(serverAddress)) { // server addr has changed..
                            serverAddress = addr;
                            publicKey = null;
                        }
                    }
                }
                
                try {
                    int k = 0;
                    byte[] msg = new byte[(int)(receiveData[k] & 0xff)];
                    k++;
                    for (int i = 0; i < msg.length && k < BROADCAST_MESSAGE_LENGTH; i++, k++) {
                        msg[i] = receiveData[k];
                    }
                    commentsTextArea.setText(new String(msg, "UTF-8"));

                    if (publicKey == null) {
                        byte[] byteMod = new byte[(int)(receiveData[k] & 0xff)];
                        k++;
                        for (int i = 0; i < byteMod.length && k < BROADCAST_MESSAGE_LENGTH; i++, k++) {
                            byteMod[i] = receiveData[k];
                        }
                        // hexDump(byteMod) ;
                        byte[] byteExp = new byte[(int)(receiveData[k] & 0xff)];
                        k++;
                        for (int i = 0; i < byteExp.length && k < BROADCAST_MESSAGE_LENGTH; i++, k++) {
                            byteExp[i] = receiveData[k];
                        }
                        // hexDump(byteExp) ;
                        regeneratePublicKey(byteMod, byteExp);
                        
                    }
                    submitButton.setEnabled(true) ;
                } catch (UnsupportedEncodingException ex) {;
                }

                try {
                    sleep(1000);
                } catch (InterruptedException ex) {
                    terminatingListener=true ;
                }
            }
        }
    }
}
