/*
 * The MIT License
 *
 * Copyright 2013-2016 Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package turtleedit;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.OutputStreamWriter;
import static java.lang.Thread.sleep;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.text.AbstractDocument;
import javax.swing.text.AttributeSet;
import javax.swing.text.BoxView;
import javax.swing.text.ComponentView;
import javax.swing.text.DefaultCaret;
import javax.swing.text.DocumentFilter;
import javax.swing.text.Element;
import javax.swing.text.IconView;
import javax.swing.text.JTextComponent;
import javax.swing.text.LabelView;
import javax.swing.text.ParagraphView;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledEditorKit;
import javax.swing.text.TabSet;
import javax.swing.text.View;
import javax.swing.text.ViewFactory;
/**
 *
 * @author Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>
 */
public class TEConsolePane extends javax.swing.JTextPane {

    public long lastFocusedTime = 0 ;
    
    ReadOutThread readThr ;
    OutputStreamWriter outWriter ;
    String charCode="UTF-8" ;
    DummyDocumentFilter dmyDocFilter ;
    Process runningProcess=null ;
    boolean silentMode ;
    boolean terminateRdr ;
    boolean runningRdr = false ;
    private final Object swingIsReadyLock = new Object() ;
    boolean swingIsReady = false ;
    TEMainWindow controller=null ;
    
    public int TAB_WIDTH = 36 ;
    
    TEConsolePane() {
        readThr = null ;
        outWriter = null ;
        
        setEditorKit(new ConsoleEditorKit());
        
        setCaret(new ConsoleCaret());
        
        getCaret().setVisible(true);
        // initKeyListener() ;
        Font consolefont = new Font(Font.MONOSPACED, Font.PLAIN, 12) ;
        setFont(consolefont) ;
        Canvas dummyCanvas = new Canvas() ;
        FontMetrics fm = dummyCanvas.getFontMetrics(consolefont);
        TAB_WIDTH = (int) (fm.stringWidth("0") * 4.0);

        setEditable(true) ;
        
        Document doc = getDocument() ;
        dmyDocFilter = new DummyDocumentFilter() ;
        ((AbstractDocument) doc).setDocumentFilter(dmyDocFilter) ;
        
        addFocusListener(new FocusListener() {
            public void focusLost(FocusEvent fe) {
                lastFocusedTime = System.currentTimeMillis() ;
                // setEditable(true);
            }
            public void focusGained(FocusEvent fe) {
                lastFocusedTime = System.currentTimeMillis() ;
                // setEditable(false);
            }
        });
        
        getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_D, KeyEvent.CTRL_MASK),"none");
        getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_E, KeyEvent.CTRL_MASK),"none");
        initKeyListener();
    }

    public void setController(TEMainWindow win) {
        controller = win ;
    }
    
    private void invokeDummyEvent() {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                synchronized(swingIsReadyLock) {
                    swingIsReady = true ;
                }
            }
        });
    }

    private void initKeyListener() {
        KeyListener keyListener = new KeyListener() {
            public void keyPressed(KeyEvent ev) {
                ;
            }

            public void keyReleased(KeyEvent ev) {
                ;
            }

            public void keyTyped(KeyEvent ev) {
                char c = ev.getKeyChar();
                if (outWriter != null) {
                    /* echo back user input */
                    // Document doc = getDocument();
                    // try {
                    // if (String.valueOf(c).getBytes().length < 2) { // if HANKAKU
                    //        doc.insertString(doc.getLength(), String.valueOf(c), null);
                    //        setCaretPosition(doc.getLength());
                    // }
                    //} catch (BadLocationException ex) {
                    //     ;
                    // }
                    try {
                        if (c == 0x04 || c == 0x08) { // CTRL-D or BS pressed
                            outWriter.write(c);
                            outWriter.flush();
                        }
                        if (c == 0x05) { // CTRL-E pressed
                           outWriter.flush();
                           outWriter.close();
                           outWriter = null;
                        }                       
                    } catch (IOException ex) {
                        ;
                    }
                }
            }
        };
        addKeyListener(keyListener);
    }
     
    public synchronized void clearAndSetText(String text) {
        dmyDocFilter.setMutable(true);
        setText(text);
        dmyDocFilter.setMutable(false);
    }

    public synchronized void appendText(String text) {
        Document doc = getDocument();
        dmyDocFilter.setMutable(true);
        try {
            doc.insertString(doc.getLength(), text, null);
            setCaretPosition(doc.getLength());
        } catch (BadLocationException ex) {
            ;
        }
        dmyDocFilter.setMutable(false);
    }
    
    public void handleProcess(Process proc, String code, boolean sflag) {
        runningProcess = proc ;
        charCode = code ;
        silentMode = sflag ;
        readThr = new ReadOutThread(proc) ; 
        terminateRdr = false ;
        try { 
            outWriter = new OutputStreamWriter(proc.getOutputStream(),charCode) ;
        } catch (UnsupportedEncodingException ex) {
            outWriter = new OutputStreamWriter(proc.getOutputStream()) ;
        }
        readThr.start() ;
        runningRdr = true ;
    }

    public void terminateRunningThread() {
        if (readThr != null && readThr.isAlive()) {
            silentMode = true; // force silent
            terminateRdr = true;
        }

        // to avoid a deadlock the following line is necessary
        // if coming to this part before the dummy event (invokeDummyEvent()) has been processed
        // ReadOutThread may not reach to the part to check terminateRdr in its main loop,
        // because event queue for the dummy event will not be processed until this method is done.
        synchronized (swingIsReadyLock) {
            swingIsReady = true;
        }

        while (runningRdr) {
            try {  
                sleep(0,1000) ;
            } catch (InterruptedException ex) {
                ;
            }
        }
        runningProcess = null ;
        outWriter = null ;
    }
    
    public void writeStringToRunningProcess(String text, boolean closeFlag) {
        if (outWriter != null) {
            try {
                outWriter.write(text,0,text.length()) ;
                if (closeFlag) {
                    outWriter.close();
                    outWriter = null ;
                }
                else {
                    outWriter.flush() ;
                }
            } catch (IOException ex) {
                ;
            }
        }
    }
    
    class ReadOutThread extends Thread {
        private static final int MAX_LENGTH = 65536 * 16 ; /* 1M bytes */  
        private static final int CBUFLEN = 256 ;
        private InputStreamReader rdr ;
        private Process assocProc ;
        private int length ;
        public ReadOutThread(Process proc) {
            try {
                rdr = new InputStreamReader(proc.getInputStream(),charCode) ;
            } catch (UnsupportedEncodingException ex) {
                rdr = new InputStreamReader(proc.getInputStream()) ;
            }
            assocProc=proc ;
            length=0 ;
        }

        @Override
        public void run() {
            // Document doc = getStyledDocument();
            Document doc = getDocument();
            int c=0 ;
            int tick=1 ;
            char[] cbuf = new char[CBUFLEN] ;
            try {
                for (;;) {
                    if (terminateRdr) {
                        break;
                    }
                    
                    if (rdr.ready()) {
                        // c = rdr.read();
                        c = rdr.read(cbuf,0,CBUFLEN) ;
                    } else {
                        try {
                            assocProc.exitValue();
                        } catch (IllegalThreadStateException e) {
                            // process is running 
                            sleep(0,1000) ;
                            continue ;
                        }
                        // process seems to be terminated
                        break ;
                    }
                                    
                    if (c <= 0) {
                        break;
                    }
                    if (length < MAX_LENGTH) {
                        /*
                        a simple trick to avoid the hanging of JTextPane
                        first invoke a dummy event into the the system queue
                        if swing is too busy, it will take a time until the event is processed                      
                        */
                        synchronized (swingIsReadyLock) {
                            swingIsReady = false;
                        }
                        invokeDummyEvent() ;
                        tick = tick/2>0 ? tick/2 : 1 ;
                        // wait until the dummy event is done by increasing waiting time
                        while (!swingIsReady) {
                            sleep(tick) ;
                            tick *= 2 ;
                            if (tick>160) tick=160 ;
                        }
                        length += c;
                        if (c == CBUFLEN) {
                            appendText(String.valueOf(cbuf));
                        } else {
                            for (int i = 0; i < c; i++) {
                                appendText(String.valueOf(cbuf[i]));
                            }
                        }
                    } else {
                        appendText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("TOO MUCH OUTPUT"));
                        break;
                    }
                }
            } catch (IOException e) {
                ;
            } catch (InterruptedException ex) {
                ;
            } 
            
            try {
                rdr.close();
                if (outWriter != null) {
                    outWriter.close();
                }
            } catch (IOException ex) {
                ;
            }
            outWriter = null; /* reset output stream */
                       
            try {
                int code=-1 ;             
                try {
                    code = assocProc.exitValue();
                } catch (IllegalThreadStateException e) {
                    assocProc.destroy() ;
                }
                sleep(0,1000) ;
                if (!silentMode) {
                    if (code == 0) {
                        appendText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("TERMINATED NORMALLY"));
                    } else {
                        appendText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("EXITED WITH CODE") + code + "\n");                      
                    }
                }
            } catch (IllegalThreadStateException e) {
                ;
            } catch (InterruptedException ex) {
                ;
            } finally {
                setCaretPosition(doc.getLength()) ;
            }   
            runningRdr = false ;
        }
    }
    
    class DummyDocumentFilter extends DocumentFilter {
        boolean mutable ;
        DummyDocumentFilter() {
            mutable = false ;
        }

        @Override
        public synchronized void insertString(DocumentFilter.FilterBypass fb, int offset, String string, AttributeSet attr) throws BadLocationException {
            if (outWriter !=null || mutable) super.insertString(fb, offset, string, attr);
            else super.insertString(fb, offset, "", attr);
        }

        /* replace (NOT insertString) is called when key pressed */
        @Override
        public void replace(DocumentFilter.FilterBypass fb, int offset, int length, String string, AttributeSet attrs) throws BadLocationException {
            if (outWriter != null) {
                    try {
                        outWriter.write(string);
                        outWriter.flush() ;
                    } catch (IOException ex) { ; }
                    super.replace(fb, offset, length, string, attrs);
            } else if (mutable) super.replace(fb, offset, length, string, attrs);
            else super.replace(fb, offset, 0, "", attrs);
        }

        @Override
        public void remove(DocumentFilter.FilterBypass fb, int offset, int length) throws BadLocationException {
            if (outWriter !=null || mutable) super.remove(fb, offset, length);
            else super.remove(fb, offset, 0);
        }
        
        public void setMutable(boolean flag) {
            mutable = flag ;
        }
    }
    
  // modified code from
  // http://www.java2s.com/Code/Java/Swing-JFC/Fanciercustomcaretclass.htm
  class ConsoleCaret extends DefaultCaret {
      protected synchronized void damage(Rectangle r) {
          if (r == null) {
              return;
          }
          x = r.x;
          y = r.y;
          height = r.height;
          if (width <= 0) {
              width = getComponent().getWidth();
          }
          repaint(); 
      }

      public void paint(Graphics g) {
          JTextComponent comp = getComponent();
          if (comp == null) {
              return;
          }

          int dot = getDot();
          Rectangle r = null;
          char dotChar;
          try {
              r = comp.modelToView(dot);
              if (r == null) {
                  return;
              }
              dotChar = comp.getText(dot, 1).charAt(0);
          } catch (BadLocationException e) {
              return;
          }

          if ((x != r.x) || (y != r.y)) {
              repaint(); 
              x = r.x; 
              y = r.y;
              height = r.height;
          }

          // g.setColor(comp.getCaretColor());
          // g.setXORMode(comp.getBackground()); 

          if (dotChar == '\n') {
              if (isVisible()) {
                  g.setColor(Color.RED) ;
                  g.fillRect(r.x, r.y, width, r.height);
              }
              width = r.height/2 ;
              return;
          }

          if (dotChar == '\t') {
              try {
                  Rectangle nextr = comp.modelToView(dot + 1);
                  if ((r.y == nextr.y) && (r.x < nextr.x)) {
                      width = nextr.x - r.x;
                      if (isVisible()) {
                          g.setColor(Color.PINK) ;
                          g.fillRect(r.x, r.y, width, r.height);
                      }
                      return;
                  } else {
                      dotChar = ' ';
                  }
              } catch (BadLocationException e) {
                  dotChar = ' ';
              }
          }

          width = g.getFontMetrics().charWidth(dotChar);
          // width = 2 ;
          if (isVisible()) {
              g.setColor(Color.RED) ;
              g.drawRect(r.x, r.y, width-1, r.height-1);
              // g.fillRect(r.x, r.y, width, r.height);
          }
      }
  }
  
      class ConsoleEditorKit extends StyledEditorKit {
        @Override
        public ViewFactory getViewFactory() {
            return new ConsoleViewFactory();
        }
    }

    class ConsoleViewFactory implements ViewFactory {
        @Override
        public View create(Element elem) {
            String kind = elem.getName();
            if (kind != null) {
                if (kind.equals(AbstractDocument.ContentElementName)) {
                    return new LabelView(elem);
                } else if (kind.equals(AbstractDocument.ParagraphElementName)) {
                    return new TEConsolePane.ConsoleParagraphView(elem);
                } else if (kind.equals(AbstractDocument.SectionElementName)) {
                    return new BoxView(elem, View.Y_AXIS);
                } else if (kind.equals(StyleConstants.ComponentElementName)) {
                    return new ComponentView(elem);
                } else if (kind.equals(StyleConstants.IconElementName)) {
                    return new IconView(elem);
                }
            }
            // default to text display
            return new LabelView(elem);
        }
    }

    class ConsoleParagraphView extends ParagraphView {
        
        public ConsoleParagraphView(Element e) {
            super(e);
        }
        
        @Override
        public float nextTabStop(float x, int tabOffset) {
            TabSet tabs = getTabSet();
            if (tabs == null) {
                return (float) (getTabBase() + (((int)( (x-getTabBase()) / TAB_WIDTH) + 1) * TAB_WIDTH));
            }
            return super.nextTabStop(x, tabOffset);
        }
  }
}
