/*
 * The MIT License
 *
 * Copyright 2013-2016 Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package turtleedit;

import java.awt.*;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.UnsupportedEncodingException;
import java.util.Stack;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.*;
import javax.swing.text.StyleConstants;
import javax.swing.undo.UndoManager;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 *
 * @author Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>
 */
public class TEEditorPane extends javax.swing.JEditorPane {

    public long lastFocusedTime = 0;
    
    private UndoManager undoMgr = new UndoManager();
    private boolean contentIsModified;
    private final String TAB = "    "; /* TAB = 4 spaces */

    private int findIndex;
    private String lastFoundStr;
    private boolean lastTimeIgnoreCase;
    private Font lineNumFont ;
    private int lineNumFontDescent ;
    public short LINE_NUMBER_WIDTH = 30;
    public int TAB_WIDTH = 36 ;
    public int prevPairedParenthesesPosition=-1 ;
    public int nextPairedParenthesesPosition=-1 ;
    
    TEEditorPane() {
        this.contentIsModified = false;
        
        setContentType("text/plain");
        setEditorKit(new NumberedEditorKit());
        // System.err.println(getEditorKitClassNameForContentType("text/plain"));
        // System.err.println(getContentType());

        setCaret(new EditorCaret()) ;
        getCaret().setBlinkRate(800);
        this.addCaretListener(new CaretListener(){
            public void caretUpdate(CaretEvent e) {
                char prevChar=0, nextChar=0 ;
                int dot = e.getDot(); 
                checkPairOfParentheses(dot);              
            }
        }) ;

        putClientProperty(TEEditorPane.HONOR_DISPLAY_PROPERTIES, Boolean.TRUE);
        setFont(new Font("Monospaced", Font.PLAIN, 14));
        lineNumFont = new Font("Monospaced", Font.PLAIN, 14) ;
        
        findIndex = 0;
        lastTimeIgnoreCase = false;

        Document doc = getDocument();

        doc.addUndoableEditListener(new UndoableEditListener() {
            @Override
            public void undoableEditHappened(UndoableEditEvent ev) {
                undoMgr.addEdit(ev.getEdit());
                contentIsModified = true;
            }
        });

        doc.addDocumentListener(new DocumentListener() {
            public void insertUpdate(DocumentEvent ev) {
                contentIsModified = true;
            }

            public void removeUpdate(DocumentEvent ev) {
                contentIsModified = true;
            }

            public void changedUpdate(DocumentEvent ev) {
                ;
            }
        });

        addFocusListener(new FocusListener() {
            public void focusLost(FocusEvent fe) {
                lastFocusedTime = System.currentTimeMillis();
            }

            public void focusGained(FocusEvent fe) {
                lastFocusedTime = System.currentTimeMillis();
            }
        });
              
        PopupMenuMouseListener popup = new PopupMenuMouseListener() ;
        this.addMouseListener(popup);
        
        // if (doc instanceof PlainDocument) {
        //    doc.putProperty(PlainDocument.tabSizeAttribute, 4);
        // } 

        // converting tab to spaces
//        ((AbstractDocument) doc).setDocumentFilter(
//                new DocumentFilter() {
//            @Override
//            public void insertString(FilterBypass fb, int offset, String string, AttributeSet attr) throws BadLocationException {
//                string = string.replaceAll("\t", TAB);
//                super.insertString(fb, offset, string, attr);
//            }
//
//            @Override
//            public void replace(FilterBypass fb, int offset, int length, String text, AttributeSet attrs) throws BadLocationException {
//                text = text.replaceAll("\t", TAB);
//                super.replace(fb, offset, length, text, attrs);
//            }
//        });

    }

    
    public void setTextFont(Font font) {
        setFont(font) ;
        Canvas dummyCanvas = new Canvas() ;
        FontMetrics fm = dummyCanvas.getFontMetrics(font);
        TAB_WIDTH = (int) (fm.stringWidth("0") * 4.0);
    }
    
    public void setLineNumFont(Font font) {
        lineNumFont=font ;
        // width 30 pixels at font size 14
        Canvas dummyCanvas = new Canvas() ;
        FontMetrics fm = dummyCanvas.getFontMetrics(font);
        int width = fm.stringWidth("0");
        lineNumFontDescent = fm.getDescent() ;
        LINE_NUMBER_WIDTH = (short) ( width * 4.0 * 1.15 ) ;
    }
    
    public void resetUndoManager() {
        undoMgr.discardAllEdits();
    }
            
    public void undo() {
        if (undoMgr.canUndo()) {
            undoMgr.undo();
        }
    }

    public void redo() {
        if (undoMgr.canRedo()) {
            undoMgr.redo();
        }
    }

    public void pasteString(String str) {
        int start = this.getSelectionStart();
        int end = this.getSelectionEnd();      
        Document doc = getDocument();
        try {
            if (end-start>0) {
                doc.remove(start,end) ;
                doc.insertString(start, str, null);
            } else {
                doc.insertString(getCaretPosition(), str, null);
            }
        } catch (BadLocationException ex) {
            ;
        }
    }
    
    public void contentIsSaved() {
        contentIsModified = false;
        // keep undo info after saving 2014-JUN-13
        // resetUndoManager() ;
    }

    public boolean isContentModified() {
        return contentIsModified;
    }

    private boolean isMatchedStringInDocument(Document doc, int index, String str, boolean ignoreCase) {
        try {
            String match = doc.getText(index, str.length());
            if (ignoreCase ? str.equalsIgnoreCase(match) : str.equals(match)) {
                javax.swing.text.DefaultHighlighter.DefaultHighlightPainter highlightPainter
                        = new javax.swing.text.DefaultHighlighter.DefaultHighlightPainter(Color.YELLOW);
                getHighlighter().addHighlight(index, index + str.length(), highlightPainter);
                findIndex = index;
                lastFoundStr = str;
                lastTimeIgnoreCase = ignoreCase;
                return true;
            }
        } catch (BadLocationException ex) {
            ex.printStackTrace();
        }
        return false ;
    }
    
    public boolean matchAndHilightString(String str, boolean ignoreCase, boolean searchInReverseDirection) {
        Document doc = getDocument();
        if (!searchInReverseDirection) {
            for (int i = findIndex; i <= doc.getLength() - str.length(); i++) {
                if (isMatchedStringInDocument(doc, i, str, ignoreCase)) {
                    setCaretPosition(i + str.length());
                    return true;
                }
            }
        } else {
            for (int i = findIndex; i >= 0; i--) {
                if (isMatchedStringInDocument(doc, i, str, ignoreCase)) {
                    setCaretPosition(i);
                    return true;

                }
            }
        }
        lastFoundStr = "";
        lastTimeIgnoreCase = ignoreCase ;
        removeHighlight();
        return false;
    }

    public void removeHighlight() {
        getHighlighter().removeAllHighlights();
    }

    public boolean findString(String findStr, boolean ignoreCase) {
        findIndex = 0;
        removeHighlight();
        return matchAndHilightString(findStr, ignoreCase, false);
    }

    public boolean findNextString() {
        boolean found=false ;
        removeHighlight();
        if (lastFoundStr == null || lastFoundStr.length() == 0) {
            return false;
        }
        findIndex = findIndex + lastFoundStr.length();
        if (findIndex > getDocument().getLength()-lastFoundStr.length()) {
            findIndex=0 ;
        } else { // search remaining part
            String str = lastFoundStr ;
            found = matchAndHilightString(lastFoundStr, lastTimeIgnoreCase, false);
            if (found) return true ;
            findIndex=0 ;
            lastFoundStr=str ;
        }
        // search from the top
        return matchAndHilightString(lastFoundStr, lastTimeIgnoreCase, false) ;
    }

   public boolean findPreviousString() {
        boolean found ;
        removeHighlight();
        if (lastFoundStr == null || lastFoundStr.length() == 0) {
            return false;
        }
        findIndex = findIndex - lastFoundStr.length();
        if (findIndex < 0) {
            findIndex = getDocument().getLength()-lastFoundStr.length() ;
        } else {
            String str=lastFoundStr ;
            found = matchAndHilightString(lastFoundStr, lastTimeIgnoreCase, true);
            if (found) return true ;
            lastFoundStr = str ;
            findIndex=getDocument().getLength()-lastFoundStr.length() ;
        }
        // search from the buttom
        return matchAndHilightString(lastFoundStr, lastTimeIgnoreCase, true);
    }
   
   public void checkPairOfParentheses(int pos) {
       Document doc = getDocument();
       String text = "";
       try {
           text = doc.getText(0, doc.getLength());
       } catch (BadLocationException ex) {
           return;
       }
       Stack<Character> stack = new Stack<Character>();
       boolean toBeRepainted=false ;
       boolean unbalanced=false ;

       if (pos > 0 && (text.charAt(pos-1) == ')' || text.charAt(pos - 1) == '}' || text.charAt(pos - 1) == ']')) {
           for (int i = pos - 1; i >= 0; i--) {
               char c = text.charAt(i);
               char c2;
               switch (c) {
                   case ')':
                   case '}':
                   case ']':
                       stack.push(c);
                       break;
                   case '(':
                   case '{':
                   case '[':
                       c2 = stack.pop();
                       if (c2 == ')' && c != '(' || c2 == '}' && c != '{' || c2 == ']' && c != '[') {
                           unbalanced = true;
                       }
                       break;
               }
               if (unbalanced) {
                   prevPairedParenthesesPosition = -1;
                   break;
               } else if (stack.empty()) {
                   prevPairedParenthesesPosition = i;
                   toBeRepainted = true;
                   break;
               }
           }
           if (!stack.empty()) {
               if (prevPairedParenthesesPosition>=0) toBeRepainted=true ;
               prevPairedParenthesesPosition = -1;
           }
       } else {
           if (prevPairedParenthesesPosition>=0) toBeRepainted=true ;
           prevPairedParenthesesPosition = -1 ;
       }
       if (text.length()>pos && (text.charAt(pos) == '(' || text.charAt(pos) == '{' || text.charAt(pos) == '[')) {
           unbalanced = false;
           stack.clear();
           for (int i = pos; i < text.length(); i++) {
               char c = text.charAt(i);
               char c2;
               switch (c) {
                   case '(':
                   case '{':
                   case '[':
                       stack.push(c);
                       break;
                   case ')':
                   case '}':
                   case ']':
                       c2 = stack.pop();
                       if (c2 == '(' && c != ')' || c2 == '{' && c != '}' || c2 == '[' && c != ']') {
                           unbalanced = true;
                       }
                       break;
               }
               if (unbalanced) {
                   nextPairedParenthesesPosition = -1;
                   break;
               } else if (stack.empty()) {
                   nextPairedParenthesesPosition = i;
                   toBeRepainted = true;
                   break;
               }
           }
           if (!stack.empty()) {
               if (nextPairedParenthesesPosition>=0) toBeRepainted=true ;
               nextPairedParenthesesPosition = -1;
           }
       } else {
           if (nextPairedParenthesesPosition>=0) toBeRepainted=true ;
           nextPairedParenthesesPosition = -1 ;
       }
       if (toBeRepainted) repaint() ;
   }
   
    /* this portion of code was taken from
     *  http://www.developer.com/java/other/article.php/3318421/Add-Line-Numbering-in-the-JEditorPane.htm
     */
    
    class NumberedEditorKit extends StyledEditorKit {

        @Override
        public ViewFactory getViewFactory() {
            return new NumberedViewFactory();
        }
    }

    class NumberedViewFactory implements ViewFactory {

        @Override
        public View create(Element elem) {
            String kind = elem.getName();
            if (kind != null) {
                if (kind.equals(AbstractDocument.ContentElementName)) {
                    return new TELabelView(elem);
                } else if (kind.equals(AbstractDocument.ParagraphElementName)) {
                    return new NumberedParagraphView(elem);
                } else if (kind.equals(AbstractDocument.SectionElementName)) {
                    return new BoxView(elem, View.Y_AXIS);
                } else if (kind.equals(StyleConstants.ComponentElementName)) {
                    return new ComponentView(elem);
                } else if (kind.equals(StyleConstants.IconElementName)) {
                    return new IconView(elem);
                }
            }
            // default to text display
            return new LabelView(elem);
        }
    }

    class NumberedParagraphView extends ParagraphView {

        private final Color pen_col = new Color(255, 200, 0);

        public NumberedParagraphView(Element e) {
            super(e);
            short top = 0;
            short left = 0;
            short bottom = 0;
            short right = 0;
            this.setInsets(top, left, bottom, right);
        }

        protected void setInsets(short top, short left, short bottom,
                short right) {
            super.setInsets((short) (top + 1),
                    (short) (left + LINE_NUMBER_WIDTH),
                    (short) (bottom + 1), right);
        }

        @Override
        protected short getTopInset() {
            return 1 ;
        }
        
        @Override
        protected short getBottomInset() {
            return 1 ;
        }
        
        @Override
        protected short getLeftInset() {
            return LINE_NUMBER_WIDTH ;
        }
        
        @Override
        public void paintChild(Graphics g, Rectangle r, int n) {
            super.paintChild(g, r, n);
            int previousLineCount = getPreviousLineCount();
            int x = r.x - getLeftInset();
            int y = r.y + r.height - lineNumFontDescent ; 
            Graphics2D g2 = (Graphics2D)g;
            g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, 
                      RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            g2.setFont(lineNumFont);
            g2.setColor(Color.blue);
            // g.drawString(String.format("%1$03d", previousLineCount + n + 1), x, y);
            if (n==0)
                g2.drawString(String.format("%04d",(previousLineCount+1)%10000), x, y);
            else
                g2.drawString(String.format("   >"), x, y);
        }

        public int getPreviousLineCount() {
            int lineCount = 0;
            View parent = this.getParent();
            int count = parent.getViewCount();
            for (int i = 0; i < count; i++) {
                if (parent.getView(i) == this) {
                    break;
                } else {
                    // lineCount += parent.getView(i).getViewCount();
                    lineCount += 1 ;
                }
            }
            return lineCount;
        }

        @Override
        public float nextTabStop(float x, int tabOffset) {
            TabSet tabs = getTabSet();
            if (tabs == null) {
                return (float) (getTabBase() + (((int)( (x-getTabBase()) / TAB_WIDTH) + 1) * TAB_WIDTH));
            }
            return super.nextTabStop(x, tabOffset);
        }

        
        // peventing line wrapping
//        public void layout(int width, int height) {
//            super.layout(Short.MAX_VALUE, height);
//        }
        
//        
//        public float getMinimumSpan(int axis) {
//            if (axis==0) 
//                return super.getPreferredSpan(axis) + NUMBERS_WIDTH ;
//            else
//                return super.getPreferredSpan(axis)  ;
//        }
    }

    class TELabelView extends LabelView {

        private final Color pen_col = new Color(255, 200, 0);
        private final BasicStroke dashed =
                new BasicStroke(1.0f, BasicStroke.CAP_BUTT,
                BasicStroke.JOIN_MITER, 10.0f, new float[]{1.0f}, 0.0f);
        private final BasicStroke solid = new BasicStroke(2.0f);

        public TELabelView(Element elem) {
            super(elem);
        }

        @Override
        public void paint(Graphics g, Shape a) {
            paintCustomLine(g, a);
            super.paint(g, a);
        }

        private void paintCustomLine(Graphics g, Shape a) {
            Rectangle rect = (a instanceof Rectangle) ? (Rectangle) a : a.getBounds();
            final int x0 = rect.x;
            final int y0 = rect.y;
            final int height = rect.height;
            final Color defaultColor = g.getColor();
            Graphics2D g2 = (Graphics2D) g;
                       
            char prevChar=0 ;
            if (getStartOffset()>0) {
                prevChar = getText(getStartOffset()-1,getStartOffset()).toString().charAt(0) ;
            }
          
            String text = getText(getStartOffset(), getEndOffset()).toString();
            FontMetrics fm = g.getFontMetrics();
            int space = 0;

            for (int i = 0; i < text.length(); i++) {
                int x = x0 + space;
                int y = y0;

                char c = text.charAt(i);
                int width = fm.charWidth(c);

                try {
                    if (String.valueOf(c).getBytes("UTF-8").length >= 2) { /* if ZENKAKU */
                        g2.setStroke(dashed);
                        g2.setPaint(pen_col);
                        g2.drawRect(x + 0, y - 1, width - 2, height - 1);
                    }
                } catch (UnsupportedEncodingException ex) {
                    ;
                }
                
                if (getStartOffset()+i == prevPairedParenthesesPosition
                        || getStartOffset()+i == nextPairedParenthesesPosition) {
                    g2.setStroke(solid);
                    g2.setPaint(pen_col);
                    g2.drawRect(x + 0, y, width , height - 1);                 
                }
                /*
                if ((i>0 && text.charAt(i-1)=='\t')
                   || (i==0 && c=='\t')) {
                        g2.setStroke(dashed);
                        g2.setPaint(pen_col);
                        g2.drawLine(x, y, x, y+height);
                }
                */
                if (c == '\t') {
                    if (prevChar=='\t') {
                        g2.setStroke(dashed);
                        g2.setPaint(pen_col);
                        g2.drawLine(x, y, x, y+height);
                    }
                    space += (int) getTabExpander().nextTabStop((float) x, i) - x;
                } else if (c == '\n') {
                    g2.setPaint(pen_col);
                    int h = rect.height;
                    int xc[] = {x + 4, x + 10, x + 7};
                    int yc[] = {y + h / 2, y + h / 2, y + h / 2 + 5};
                    g2.fillPolygon(xc, yc, 3);
                } else if (c < ' ') {
                    ; // skip ctrl code
                } else {
                    space += width;
                }
                prevChar = c ;
            }
            g.setColor(defaultColor);
        }
    }
 
    
    class EditorCaret extends DefaultCaret {

        protected synchronized void damage(Rectangle r) {
            if (r == null) {
                return;
            }
            x = r.x;
            y = r.y;
            height = r.height;
            if (width <= 0) {
                width = getComponent().getWidth();
            }
            repaint();
        }

        public void paint(Graphics g) {
            JTextComponent comp = getComponent();
            if (comp == null) {
                return;
            }

            int dot = getDot();
            Rectangle r = null;
            // char prevChar;
            try {
                r = comp.modelToView(dot);
                if (r == null) {
                    return;
                }
                // if (dot>0) {
                //    prevChar = comp.getText(dot-1, 1).charAt(0);
                //    if (prevChar==')' || prevChar=='}' || prevChar==']') checkPairOfParentheses(dot-1) ;
                //    else pairedParenthesesPosition=-1 ;
                // }
            } catch (BadLocationException e) {
                return;
            }

            if ((x != r.x) || (y != r.y)) {
                repaint();
                x = r.x;
                y = r.y;
                height = r.height;
            }
            // g.setXORMode(comp.getBackground());
            // width = g.getFontMetrics().charWidth(dotChar);
            width = 2;
            if (isVisible()) {
                g.setColor(Color.GRAY);
                g.fillRect(r.x, r.y, width, r.height);
            }
        }
    }
    
    
    // Reference: http://www.ne.jp/asahi/hishidama/home/tech/java/swing/JPopupMenu.html
    class PopupMenuMouseListener implements MouseListener {

        public void mouseClicked(MouseEvent e) { }

        public void mousePressed(MouseEvent e) {
            mousePopup(e);
        }

        public void mouseReleased(MouseEvent e) {
            mousePopup(e);
        }

        public void mouseEntered(MouseEvent e) { }

        public void mouseExited(MouseEvent e) { }

        private void mousePopup(MouseEvent e) {
            if (e.isPopupTrigger()) {
                JComponent c = (JComponent) e.getSource();
                showPopup(c, e.getX(), e.getY());
                e.consume();
            }
        }

        protected void showPopup(JComponent c, int x, int y) {
            JPopupMenu pmenu = new JPopupMenu();

            ActionMap am = c.getActionMap();

            Action cut = am.get(DefaultEditorKit.cutAction);
            addMenu(pmenu, java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("CUT"), cut, 'X', KeyStroke.getKeyStroke(KeyEvent.VK_X, KeyEvent.CTRL_DOWN_MASK));

            Action copy = am.get(DefaultEditorKit.copyAction);
            addMenu(pmenu, java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("COPY"), copy, 'C', KeyStroke.getKeyStroke(KeyEvent.VK_C, KeyEvent.CTRL_DOWN_MASK));

            Action paste = am.get(DefaultEditorKit.pasteAction);
            addMenu(pmenu, java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("PASTE"), paste, 'V', KeyStroke.getKeyStroke(KeyEvent.VK_V, KeyEvent.CTRL_DOWN_MASK));

            pmenu.show(c, x, y);
        }

        protected void addMenu(JPopupMenu pmenu, String text, Action action, int mnemonic, KeyStroke ks) {
            if (action != null) {
                JMenuItem mi = pmenu.add(action);
                if (text != null) {
                    mi.setText(text);
                }
                if (mnemonic != 0) {
                    mi.setMnemonic(mnemonic);
                }
                if (ks != null) {
                    mi.setAccelerator(ks);
                }
            }
        }
    }

}

