/*
 * The MIT License
 *
 * Copyright 2013 Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package turtleedit;

import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import java.util.prefs.*;
import java.util.StringTokenizer;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.imageio.ImageIO;
import javax.swing.WindowConstants;
import javax.swing.filechooser.FileNameExtensionFilter;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import static java.lang.Thread.sleep;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Scanner;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.KeyStroke;
/**
 *
 * @author Yoshinori Hayakawa <hayakawa@cite.tohoku.ac.jp>
 */
public class TEMainWindow extends javax.swing.JFrame {

    private File workingFile = null ;
    private Boolean workingFileIsSet;
    private Boolean contentIsSaved;
    private Boolean startCompiling ;
    private Boolean compileWasDoneNormally;
    public Boolean interpreterMode ;
    public String compileCommand;
    public String execCommand;
    public String execPath;
    public String helpURI;
    public String charCode;
    public String preferredFontName ;
    public int preferredFontSize ;
    public String passcodeForFeedback ;
    public String nicknameForFeedback ;
    public int channelForFeedback ;
    public String turtleFieldPathName ;
    public boolean turtleFieldAutoLaunch ;
    
    TFLauncher tfLauncher ;
            
    private Preferences prefs;
    private Process runningProcess;
    private String foundString;
    
    private TEFilterDialog filterDialog ;
    private TEReplaceDialog replaceDialog ;
    private TEFindDialog findDialog ;
    private TEResponseDialog feedbackDialog ;
    private TEExecCommandDialog execCommandDialog ;
    
    private TESortDialog sortDialog ;
    private TECountDialog countDialog ;
    private TEUniqDialog uniqDialog ;
    private TETrDialog trDialog ;

    public TEMainWindow() {
        initComponents();
        workingFileIsSet = false;
        contentIsSaved = false;
        startCompiling = false ;
        compileWasDoneNormally = false;
        runningProcess = null;
        filterDialog = null ;
        replaceDialog = null ;
        findDialog = null ;
        feedbackDialog = null ;
        execCommandDialog = null ;
        sortDialog = null ;
        countDialog = null ;
        uniqDialog = null ;
        trDialog = null ;
        
        tfLauncher = null ;
        
        turtleFieldAutoLaunch = false ;
        turtleFieldPathName = "" ;
          
        Image img;
        try {
            img = ImageIO.read(getClass().getResource("images/tedit-icon.png"));
            setIconImage(img);
        } catch (IOException ex) { ; }
          
        prefs = Preferences.userNodeForPackage(this.getClass());

        restorePreferences();
        restoreWindowLocationAndSize() ;

        setMonospacedFont();

        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                exitMenuItemActionPerformed(null);
            }
        });
        
        tfLauncher = new TFLauncher(this);
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jMenu4 = new javax.swing.JMenu();
        jMenuItem1 = new javax.swing.JMenuItem();
        jMenuItem2 = new javax.swing.JMenuItem();
        jPanel1 = new javax.swing.JPanel();
        compileButton = new javax.swing.JButton();
        runButton = new javax.swing.JButton();
        breakButton = new javax.swing.JButton();
        pwdButton = new javax.swing.JButton();
        filler1 = new javax.swing.Box.Filler(new java.awt.Dimension(100, 46), new java.awt.Dimension(100, 46), new java.awt.Dimension(100, 46));
        terminalSplitPane = new javax.swing.JSplitPane();
        jScrollPane1 = new javax.swing.JScrollPane();
        editPane = new TEEditorPane() ;
        jScrollPane3 = new javax.swing.JScrollPane();
        consolePane = new TEConsolePane() ;
        jMenuBar1 = new javax.swing.JMenuBar();
        jMenu1 = new javax.swing.JMenu();
        newMenuItem = new javax.swing.JMenuItem();
        openMenuItem = new javax.swing.JMenuItem();
        saveMenuItem = new javax.swing.JMenuItem();
        saveAsMenuItem = new javax.swing.JMenuItem();
        saveOutputsMenuItem = new javax.swing.JMenuItem();
        jSeparator1 = new javax.swing.JPopupMenu.Separator();
        exitMenuItem = new javax.swing.JMenuItem();
        jMenu2 = new javax.swing.JMenu();
        undoMenuItem = new javax.swing.JMenuItem();
        cutMenuItem = new javax.swing.JMenuItem();
        copyMenuItem = new javax.swing.JMenuItem();
        pasteMenuItem = new javax.swing.JMenuItem();
        jSeparator2 = new javax.swing.JPopupMenu.Separator();
        findMenuItem = new javax.swing.JMenuItem();
        findNextMenuItem = new javax.swing.JMenuItem();
        findPreviousMenuItem = new javax.swing.JMenuItem();
        filterMenu = new javax.swing.JMenu();
        regExFilterMenuItem = new javax.swing.JMenuItem();
        regExReplaceMenuItem = new javax.swing.JMenuItem();
        sortMenuItem = new javax.swing.JMenuItem();
        uniqMenuItem = new javax.swing.JMenuItem();
        countMenuItem = new javax.swing.JMenuItem();
        translationMenuItem = new javax.swing.JMenuItem();
        jSeparator6 = new javax.swing.JPopupMenu.Separator();
        execCommandMenuItem = new javax.swing.JMenuItem();
        jSeparator5 = new javax.swing.JPopupMenu.Separator();
        swapBuffersMenuItem = new javax.swing.JMenuItem();
        jMenu3 = new javax.swing.JMenu();
        compileMenuItem = new javax.swing.JMenuItem();
        runMenuItem = new javax.swing.JMenuItem();
        breakMenuItem = new javax.swing.JMenuItem();
        jSeparator4 = new javax.swing.JPopupMenu.Separator();
        settingsMenuItem = new javax.swing.JMenuItem();
        jMenu5 = new javax.swing.JMenu();
        webHelpMenuItem = new javax.swing.JMenuItem();
        feedbackMenuItem = new javax.swing.JMenuItem();
        aboutMenuItem = new javax.swing.JMenuItem();

        jMenu4.setText("jMenu4");

        jMenuItem1.setText("jMenuItem1");

        jMenuItem2.setText("jMenuItem2");

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("TurtleEdit");
        setMinimumSize(new java.awt.Dimension(500, 300));

        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("turtleedit/TEBandle"); // NOI18N
        compileButton.setText(bundle.getString("COMPILE")); // NOI18N
        compileButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                compileButtonActionPerformed(evt);
            }
        });

        runButton.setText(bundle.getString("RUN")); // NOI18N
        runButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                runButtonActionPerformed(evt);
            }
        });

        breakButton.setText(bundle.getString("BREAK")); // NOI18N
        breakButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                breakButtonActionPerformed(evt);
            }
        });

        pwdButton.setText(bundle.getString("CURRENT DIRECTORY")); // NOI18N
        pwdButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                pwdButtonActionPerformed(evt);
            }
        });

        org.jdesktop.layout.GroupLayout jPanel1Layout = new org.jdesktop.layout.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .add(compileButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 102, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(runButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 93, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(breakButton)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(filler1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 72, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.UNRELATED, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .add(pwdButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 176, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel1Layout.createSequentialGroup()
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                    .add(pwdButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 46, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                        .add(compileButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 46, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                        .add(runButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 46, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                        .add(breakButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 46, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                    .add(filler1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addContainerGap())
        );

        terminalSplitPane.setDividerLocation(300);
        terminalSplitPane.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);

        jScrollPane1.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);

        editPane.setFont(new java.awt.Font("Monaco", 0, 14)); // NOI18N
        jScrollPane1.setViewportView(editPane);

        terminalSplitPane.setTopComponent(jScrollPane1);

        consolePane.setFont(new java.awt.Font("Monaco", 0, 12)); // NOI18N
        jScrollPane3.setViewportView(consolePane);

        terminalSplitPane.setBottomComponent(jScrollPane3);

        jMenu1.setText(bundle.getString("FILE")); // NOI18N

        newMenuItem.setText(bundle.getString("NEW")); // NOI18N
        newMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                newMenuItemActionPerformed(evt);
            }
        });
        jMenu1.add(newMenuItem);

        openMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_O, java.awt.event.InputEvent.CTRL_MASK));
        openMenuItem.setText(bundle.getString("OPEN")); // NOI18N
        openMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                openMenuItemActionPerformed(evt);
            }
        });
        jMenu1.add(openMenuItem);

        saveMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_S, java.awt.event.InputEvent.CTRL_MASK));
        saveMenuItem.setText(bundle.getString("SAVE")); // NOI18N
        saveMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveMenuItemActionPerformed(evt);
            }
        });
        jMenu1.add(saveMenuItem);

        saveAsMenuItem.setText(bundle.getString("SAVE AS...")); // NOI18N
        saveAsMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveAsMenuItemActionPerformed(evt);
            }
        });
        jMenu1.add(saveAsMenuItem);

        saveOutputsMenuItem.setText(bundle.getString("SAVE OUTPUTS")); // NOI18N
        saveOutputsMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveOutputsMenuItemActionPerformed(evt);
            }
        });
        jMenu1.add(saveOutputsMenuItem);
        jMenu1.add(jSeparator1);

        exitMenuItem.setText(bundle.getString("EXIT")); // NOI18N
        exitMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                exitMenuItemActionPerformed(evt);
            }
        });
        jMenu1.add(exitMenuItem);

        jMenuBar1.add(jMenu1);

        jMenu2.setText(bundle.getString("EDIT")); // NOI18N

        undoMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_Z, java.awt.event.InputEvent.CTRL_MASK));
        undoMenuItem.setText(bundle.getString("UNDO")); // NOI18N
        undoMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                undoMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(undoMenuItem);

        cutMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_X, java.awt.event.InputEvent.CTRL_MASK));
        cutMenuItem.setText(bundle.getString("CUT")); // NOI18N
        cutMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cutMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(cutMenuItem);

        copyMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_C, java.awt.event.InputEvent.CTRL_MASK));
        copyMenuItem.setText(bundle.getString("COPY")); // NOI18N
        copyMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                copyMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(copyMenuItem);

        pasteMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_V, java.awt.event.InputEvent.CTRL_MASK));
        pasteMenuItem.setText(bundle.getString("PASTE")); // NOI18N
        pasteMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                pasteMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(pasteMenuItem);
        jMenu2.add(jSeparator2);

        findMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_F, java.awt.event.InputEvent.CTRL_MASK));
        findMenuItem.setText(bundle.getString("FIND")); // NOI18N
        findMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                findMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(findMenuItem);

        findNextMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_G, java.awt.event.InputEvent.CTRL_MASK));
        findNextMenuItem.setText(bundle.getString("FIND NEXT")); // NOI18N
        findNextMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                findNextMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(findNextMenuItem);

        findPreviousMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_G, java.awt.event.InputEvent.SHIFT_MASK | java.awt.event.InputEvent.CTRL_MASK));
        findPreviousMenuItem.setText(bundle.getString("FIND PREVIOUS")); // NOI18N
        findPreviousMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                findPreviousMenuItemActionPerformed(evt);
            }
        });
        jMenu2.add(findPreviousMenuItem);

        jMenuBar1.add(jMenu2);

        filterMenu.setText(bundle.getString("FILTER")); // NOI18N

        regExFilterMenuItem.setText(bundle.getString("REGEX FILTER")); // NOI18N
        regExFilterMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                regExFilterMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(regExFilterMenuItem);

        regExReplaceMenuItem.setText(bundle.getString("REPLACE")); // NOI18N
        regExReplaceMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                regExReplaceMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(regExReplaceMenuItem);

        sortMenuItem.setText("Sort");
        sortMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sortMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(sortMenuItem);

        uniqMenuItem.setText("Uniq");
        uniqMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                uniqMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(uniqMenuItem);

        countMenuItem.setText("Count");
        countMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                countMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(countMenuItem);

        translationMenuItem.setText("Tr");
        translationMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                translationMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(translationMenuItem);
        filterMenu.add(jSeparator6);

        execCommandMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_4, java.awt.event.InputEvent.ALT_MASK));
        execCommandMenuItem.setText(bundle.getString("EXEC COMMAND")); // NOI18N
        execCommandMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                execCommandMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(execCommandMenuItem);
        filterMenu.add(jSeparator5);

        swapBuffersMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_5, java.awt.event.InputEvent.ALT_MASK));
        swapBuffersMenuItem.setText(bundle.getString("SWAP BUFFERS")); // NOI18N
        swapBuffersMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                swapBuffersMenuItemActionPerformed(evt);
            }
        });
        filterMenu.add(swapBuffersMenuItem);

        jMenuBar1.add(filterMenu);

        jMenu3.setText(bundle.getString("RUN")); // NOI18N

        compileMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_1, java.awt.event.InputEvent.ALT_MASK));
        compileMenuItem.setText(bundle.getString("CHECK & COMPILE")); // NOI18N
        compileMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                compileMenuItemActionPerformed(evt);
            }
        });
        jMenu3.add(compileMenuItem);

        runMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_2, java.awt.event.InputEvent.ALT_MASK));
        runMenuItem.setText(bundle.getString("RUN")); // NOI18N
        runMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                runMenuItemActionPerformed(evt);
            }
        });
        jMenu3.add(runMenuItem);

        breakMenuItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_3, java.awt.event.InputEvent.ALT_MASK));
        breakMenuItem.setText(bundle.getString("BREAK")); // NOI18N
        breakMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                breakMenuItemActionPerformed(evt);
            }
        });
        jMenu3.add(breakMenuItem);
        jMenu3.add(jSeparator4);

        settingsMenuItem.setText(bundle.getString("SETTINGS")); // NOI18N
        settingsMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                settingsMenuItemActionPerformed(evt);
            }
        });
        jMenu3.add(settingsMenuItem);

        jMenuBar1.add(jMenu3);

        jMenu5.setText(bundle.getString("HELP")); // NOI18N

        webHelpMenuItem.setText(bundle.getString("OPEN WEB PAGE")); // NOI18N
        webHelpMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                webHelpMenuItemActionPerformed(evt);
            }
        });
        jMenu5.add(webHelpMenuItem);

        feedbackMenuItem.setText(bundle.getString("FEEDBACK")); // NOI18N
        feedbackMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                feedbackMenuItemActionPerformed(evt);
            }
        });
        jMenu5.add(feedbackMenuItem);

        aboutMenuItem.setText(bundle.getString("ABOUT TURTLEEDIT...")); // NOI18N
        aboutMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                aboutMenuItemActionPerformed(evt);
            }
        });
        jMenu5.add(aboutMenuItem);

        jMenuBar1.add(jMenu5);

        setJMenuBar(jMenuBar1);

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
            .add(terminalSplitPane, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 0, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .add(jPanel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(terminalSplitPane, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 404, Short.MAX_VALUE))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void newMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_newMenuItemActionPerformed
        if (confirmToSaveDialogue() == 0) {
            return; /* cancelled */
        }
        editPane.selectAll();
        editPane.replaceSelection("");
        ((TEEditorPane)editPane).resetUndoManager() ;
        workingFileIsSet = false;
        contentIsSaved = false;
        setWindowTitle();
    }//GEN-LAST:event_newMenuItemActionPerformed

    private void exitMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_exitMenuItemActionPerformed
        if (confirmToSaveDialogue() == 0) {
            return;
        }
        if (workingFile!=null) {
            prefs.put("WORKING_DIR", workingFile.getParent());
        }
        saveWindowLocationAndSize();
        killRunningProcess(false);
        System.exit(0);
    }//GEN-LAST:event_exitMenuItemActionPerformed

    private void pasteMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_pasteMenuItemActionPerformed
        /* workaround for very slow pasting at the ICL system */
        Toolkit kit = Toolkit.getDefaultToolkit();
        Clipboard cb = kit.getSystemClipboard();
        String str;
        try {
            str = (String) cb.getData(DataFlavor.stringFlavor);

        } catch (UnsupportedFlavorException e) {
            return;
        } catch (IOException e) {
            return;
        }
        ((TEEditorPane)editPane).pasteString(str);
    }//GEN-LAST:event_pasteMenuItemActionPerformed

    private void compileMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_compileMenuItemActionPerformed
        compileWorkingFile();
    }//GEN-LAST:event_compileMenuItemActionPerformed

    private void openMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_openMenuItemActionPerformed

        if (confirmToSaveDialogue() == 0) {
            return; /* cancelled */
        }

        JFileChooser fileChooser = new JFileChooser(initialDirectoryToOpen());
        FileNameExtensionFilter filter1 = new FileNameExtensionFilter(
                java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("PROGRAMMING"),
                "c", "cpp", "java", "rb", "py", "js", "php", "pl", "bas", "lua", "scala", "clj", "groovy");
        FileNameExtensionFilter filter2 = new FileNameExtensionFilter(
                java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("WRITING"),
                "txt", "tex", "htm", "html");
        fileChooser.addChoosableFileFilter(filter2);
        fileChooser.addChoosableFileFilter(filter1);
        if (fileChooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            this.workingFile = fileChooser.getSelectedFile();
            workingFileIsSet = true;
            BufferedReader reader;
            StringBuilder stringBuilder = new StringBuilder();
            try {
                reader = new BufferedReader(
                        new InputStreamReader(new FileInputStream(this.workingFile), charCode));
                while (reader.ready()) {
                    stringBuilder.append(reader.readLine() + "\n");
                }
                reader.close();
                this.editPane.setText(stringBuilder.toString());
                contentIsSaved = false;
                ((TEEditorPane) editPane).contentIsSaved();
            } catch (IOException ioe) {
                ((TEConsolePane) consolePane).clearAndSetText("Error: cannot open " + workingFile.getName());
            }
        }
        setWindowTitle();
    }//GEN-LAST:event_openMenuItemActionPerformed

    private void saveAsMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveAsMenuItemActionPerformed
        JFileChooser fileChooser = new JFileChooser(initialDirectoryToOpen());
        if (fileChooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
            this.workingFile = fileChooser.getSelectedFile();
            if (!isFileNameOK(workingFile.getName())) {
                return;
            }
            if (workingFile.exists() && !canOverwriteFile(workingFile.getName())) {
                return;
            }
            workingFileIsSet = true;
            BufferedWriter writer;
            try {
                writer = new BufferedWriter(
                        new OutputStreamWriter(new FileOutputStream(this.workingFile), charCode));
                writer.write(this.editPane.getText());
                writer.close();
                this.workingFile = fileChooser.getSelectedFile();
                contentIsSaved = true;
                ((TEEditorPane) editPane).contentIsSaved();
            } catch (IOException ioe) {
                ((TEConsolePane) consolePane).clearAndSetText("Error: cannot save data to "+ workingFile.getName() + "\n");
            }
        }
        setWindowTitle();
    }//GEN-LAST:event_saveAsMenuItemActionPerformed

    private void findMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_findMenuItemActionPerformed
        if (findDialog == null) findDialog = new TEFindDialog(this, true);
        findDialog.setVisible(true);
    }//GEN-LAST:event_findMenuItemActionPerformed

    private void saveMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveMenuItemActionPerformed
        if (!workingFileIsSet) {
            JFileChooser fileChooser = new JFileChooser(initialDirectoryToOpen());
            if (fileChooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
                this.workingFile = fileChooser.getSelectedFile();
                if (!isFileNameOK(workingFile.getName())) {
                    return;
                }
                if (workingFile.exists() && !canOverwriteFile(workingFile.getName())) {
                    return;
                }
                workingFileIsSet = true;
            } else {
                return;
            }
            // isFileNameOK(workingFile.getName()) ;
            setWindowTitle();
        }
        BufferedWriter writer;
        try {
            writer = new BufferedWriter(
                    new OutputStreamWriter(new FileOutputStream(this.workingFile), charCode));
            writer.write(this.editPane.getText());
            writer.close();
            contentIsSaved = true;
            ((TEEditorPane) editPane).contentIsSaved();
        } catch (IOException ioe) {
            ((TEConsolePane) consolePane).clearAndSetText("Error: cannot save file¥n");
        }
        setWindowTitle() ;
    }//GEN-LAST:event_saveMenuItemActionPerformed

    private void cutMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cutMenuItemActionPerformed
        editPane.cut();
    }//GEN-LAST:event_cutMenuItemActionPerformed

    private void copyMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_copyMenuItemActionPerformed
        if ( ((TEEditorPane)editPane).lastFocusedTime  
                > ((TEConsolePane)consolePane).lastFocusedTime ) {
            editPane.copy();
        } else {
            consolePane.copy() ;
        }
    }//GEN-LAST:event_copyMenuItemActionPerformed

    private void undoMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_undoMenuItemActionPerformed
        ((TEEditorPane) editPane).undo();
    }//GEN-LAST:event_undoMenuItemActionPerformed

    private void settingsMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_settingsMenuItemActionPerformed
        TESettingDialog setting = new TESettingDialog(this, true);
        setting.setInterpreterMode(interpreterMode) ;
        setting.setCompileCommandTextField(compileCommand);
        setting.setExecCommandTextField(execCommand);
        setting.setExecPathTextField(execPath);
        setting.setCharCodeRadioButton(charCode);
        setting.setPreferredFontName(preferredFontName) ;
        setting.setPreferredFontSize(preferredFontSize) ;
        setting.setTurtleFieldPathName(turtleFieldPathName) ;
        setting.setTurtleFieldAutoLaunch(turtleFieldAutoLaunch) ;
        setting.setVisible(true);
    }//GEN-LAST:event_settingsMenuItemActionPerformed

    private void compileButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_compileButtonActionPerformed
        compileMenuItemActionPerformed(null);
    }//GEN-LAST:event_compileButtonActionPerformed

    private void runMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_runMenuItemActionPerformed
        // execCompiledCommand() ;
        execCompiledCommand();
    }//GEN-LAST:event_runMenuItemActionPerformed

    private void runButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_runButtonActionPerformed
        runMenuItemActionPerformed(null);
    }//GEN-LAST:event_runButtonActionPerformed

    private void breakButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_breakButtonActionPerformed
        killRunningProcess(false);
    }//GEN-LAST:event_breakButtonActionPerformed

    private void findNextMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_findNextMenuItemActionPerformed
        findNextString();
    }//GEN-LAST:event_findNextMenuItemActionPerformed

    private void pwdButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_pwdButtonActionPerformed
        if (!workingFileIsSet) {
            JOptionPane.showMessageDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("FILE IS NOT OPENED."),
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("WARNING"),
                    JOptionPane.WARNING_MESSAGE);
            return;
        }
        TEDirListDialog dirList = new TEDirListDialog(this, true, workingFile);
        dirList.setVisible(true);
    }//GEN-LAST:event_pwdButtonActionPerformed

    private void aboutMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_aboutMenuItemActionPerformed
        TEAboutDialog about = new TEAboutDialog(this, true);
        about.setVisible(true);
    }//GEN-LAST:event_aboutMenuItemActionPerformed

    private void webHelpMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_webHelpMenuItemActionPerformed
        // TEHelpWindow help = new TEHelpWindow(this);
        TEHelpDialog help = new TEHelpDialog(this,false);
        help.setVisible(true);
    }//GEN-LAST:event_webHelpMenuItemActionPerformed

    private void saveOutputsMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveOutputsMenuItemActionPerformed
        JFileChooser fileChooser = new JFileChooser(initialDirectoryToOpen());
        if (fileChooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
            File outputFile = fileChooser.getSelectedFile();
            if (!isFileNameOK(outputFile.getName())) {
                return;
            }
            if (outputFile.exists() && !canOverwriteFile(outputFile.getName())) {
                return;
            }
            BufferedWriter writer;
            try {
                writer = new BufferedWriter(
                        new OutputStreamWriter(new FileOutputStream(outputFile), charCode));
                writer.write(consolePane.getText());
                writer.close();
            } catch (IOException ioe) {
                ;
            }
        }
    }//GEN-LAST:event_saveOutputsMenuItemActionPerformed

    private void breakMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_breakMenuItemActionPerformed
        breakButtonActionPerformed(null) ;
    }//GEN-LAST:event_breakMenuItemActionPerformed

    private void feedbackMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_feedbackMenuItemActionPerformed
        if (feedbackDialog==null) feedbackDialog = new TEResponseDialog(this,false) ;
        feedbackDialog.setNicknamePasscodeAndChannel(nicknameForFeedback, passcodeForFeedback, channelForFeedback);
        if (feedbackDialog.openListenerSocket()) {
            feedbackDialog.startListener() ;
            feedbackDialog.setVisible(true) ;
        } 
    }//GEN-LAST:event_feedbackMenuItemActionPerformed

    private void regExFilterMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_regExFilterMenuItemActionPerformed
        if (filterDialog == null) {
            filterDialog = new TEFilterDialog(this, false);
        }
        filterDialog.setVisible(true);
    }//GEN-LAST:event_regExFilterMenuItemActionPerformed

    private void regExReplaceMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_regExReplaceMenuItemActionPerformed
        if (replaceDialog==null)  replaceDialog = new TEReplaceDialog(this,false) ;
        replaceDialog.setVisible(true) ;
    }//GEN-LAST:event_regExReplaceMenuItemActionPerformed

    private void swapBuffersMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_swapBuffersMenuItemActionPerformed
        String buffer = editPane.getText() ;
        editPane.selectAll() ;
        editPane.setText(consolePane.getText());
        ((TEConsolePane) consolePane).clearAndSetText(buffer);
    }//GEN-LAST:event_swapBuffersMenuItemActionPerformed

    private void uniqMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_uniqMenuItemActionPerformed
        if (uniqDialog == null) uniqDialog = new TEUniqDialog(this, false);
        uniqDialog.setVisible(true);
    }//GEN-LAST:event_uniqMenuItemActionPerformed

    private void sortMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sortMenuItemActionPerformed
        if (sortDialog == null) sortDialog = new TESortDialog(this, false);
        sortDialog.setVisible(true);
    }//GEN-LAST:event_sortMenuItemActionPerformed

    private void execCommandMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_execCommandMenuItemActionPerformed
        if (execCommandDialog == null) execCommandDialog = new TEExecCommandDialog(this, false);
        execCommandDialog.setVisible(true);        // TODO add your handling code here:
    }//GEN-LAST:event_execCommandMenuItemActionPerformed

    private void countMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_countMenuItemActionPerformed
        if (countDialog == null) countDialog = new TECountDialog(this, false);
        countDialog.setVisible(true);
    }//GEN-LAST:event_countMenuItemActionPerformed

    private void translationMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_translationMenuItemActionPerformed
        if (trDialog == null) trDialog = new TETrDialog(this, false);
        trDialog.setVisible(true);
    }//GEN-LAST:event_translationMenuItemActionPerformed

    private void findPreviousMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_findPreviousMenuItemActionPerformed
        findPreviousString();
    }//GEN-LAST:event_findPreviousMenuItemActionPerformed

  private void setWindowTitle() {
        if (workingFileIsSet) {
            this.setTitle("TurtleEdit: " + workingFile.getName()) ;
        } else { 
            this.setTitle("TurtleEdit") ;
        }
    }
                
    private int confirmToSaveDialogue() {
        if (((TEEditorPane) editPane).isContentModified()) {
            int rc = JOptionPane.showConfirmDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("DO YOU WANT TO SAVE THE CHANGES YOU MADE ?"),
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("CONFIRMATION MESSAGE"),
                    JOptionPane.YES_NO_CANCEL_OPTION
                    );

            if (rc == JOptionPane.YES_OPTION) {
                this.saveMenuItemActionPerformed(null);
            } else if (rc == JOptionPane.CANCEL_OPTION) {
                return 0;
            }
        }
        return 1;
    }

    
     public String initialDirectoryToOpen() {
        if (workingFile == null) {
            String lastWorkingDir = prefs.get("WORKING_DIR", "");
            if (lastWorkingDir.length()>0) {
                File workingDir = new File(lastWorkingDir) ;
                if (workingDir.exists()) {
                    try {
                        return workingDir.getCanonicalPath().replaceAll("\\\\", "\\\\\\\\") ;
                    } catch (IOException ex) {
                        return null ;
                    }
                } else {
                    return null ;
                }
            } else {
                return null;
            }
        } else {
            return workingFile.getParent().replaceAll("\\\\", "\\\\\\\\");
        }
    }
    
    public boolean isFileNameOK(String fn) {
        if (fn.contains(" ") || fn.contains("　")) {
            JOptionPane.showMessageDialog(
                    this,
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("FILE NAME")
                    + " \"" + fn + "\" " + java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("CONTAINS WHITE SPACE"),
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("WARNING"),
                    JOptionPane.WARNING_MESSAGE);
            return false;
        }
        return true;
    }

    public boolean canOverwriteFile(String fn) {
        int rc = JOptionPane.showConfirmDialog(
                this,
                java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("FILE")
                + " \"" + fn + "\" " + java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("EXIST. DO YOU WANT TO OVERWRITE IT ?"),
                java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("WARNING"),
                JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
        if (rc == JOptionPane.OK_OPTION) {
            return true;
        } else {
            return false;
        }
    }

    public void findString(String str, boolean ignoreCase) {
        foundString = null;
        if (!((TEEditorPane) editPane).findString(str, ignoreCase)) {
            JOptionPane.showMessageDialog(
                    this,
                    "\"" + str + "\"" + java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("NOT FOUND"),
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("MESSAGE"),
                    JOptionPane.INFORMATION_MESSAGE);
        } else {
            foundString = str;
        }
    }

    public void findNextString() {
        if (foundString != null
                && !((TEEditorPane) editPane).findNextString()) {
            JOptionPane.showMessageDialog(
                    this,
                    "\"" + foundString + "\" not found.",
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("MESSAGE"),
                    JOptionPane.INFORMATION_MESSAGE);
        }
    }
    
    public void findPreviousString() {
        if (foundString != null
                && !((TEEditorPane) editPane).findPreviousString()) {
            JOptionPane.showMessageDialog(
                    this,
                    "\"" + foundString + "\" not found.",
                    java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("MESSAGE"),
                    JOptionPane.INFORMATION_MESSAGE);
        }
    }

    private void restoreWindowLocationAndSize() {
        int x, y, width, height ;
        x = prefs.getInt("WIN_LOC_X", 10);
        y = prefs.getInt("WIN_LOC_Y", 10);
        width = prefs.getInt("WIN_WIDTH", 570);
        height = prefs.getInt("WIN_HEIGHT", 600);
        this.setLocation(x,y) ;
        this.setSize(width,height) ;
    }
    
    private void saveWindowLocationAndSize() {
        Rectangle fbound = this.getBounds();
        prefs.putInt("WIN_LOC_X", fbound.x);
        prefs.putInt("WIN_LOC_Y", fbound.y);
        prefs.putInt("WIN_WIDTH", fbound.width);
        prefs.putInt("WIN_HEIGHT", fbound.height);
    }
    
    public void savePreferences() {
        if (interpreterMode) prefs.put("INTERPRETER_MODE", "YES");
        else prefs.put("INTERPRETER_MODE", "NO");
        
        prefs.put("COMPILE_COMMAND", compileCommand);
        prefs.put("EXEC_COMMAND", execCommand);
        prefs.put("EXEC_PATH", execPath);
        prefs.put("HELP_URI", helpURI);
        prefs.put("CHAR_CODE", charCode);
        prefs.put("PREFERRED_FONT_NAME", preferredFontName);
        prefs.put("PREFERRED_FONT_SIZE", String.valueOf(preferredFontSize)) ;
        prefs.put("TF_PATH_NAME", turtleFieldPathName);
        if (turtleFieldAutoLaunch) prefs.put("TF_AUTO_LAUNCH", "YES");
        else prefs.put("TF_AUTO_LAUNCH", "NO");
        try {
            prefs.flush();
        } catch (java.util.prefs.BackingStoreException e) {
            ;
        }
    }

    public void restorePreferences() {
        
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("turtleedit/TEBandle");
       
        String im = prefs.get("INTERPRETER_MODE", "NO") ;
        interpreterMode = im.equals("YES") ;
        compileButton.setEnabled(!interpreterMode) ;
        compileMenuItem.setEnabled(!interpreterMode) ;
        
        compileCommand = prefs.get("COMPILE_COMMAND", "cc %f -lm");
        execCommand = prefs.get("EXEC_COMMAND", "./a.out");
        execPath = prefs.get("EXEC_PATH", "/opt/local/bin");
        helpURI = prefs.get("HELP_URI", "http://seaotter.cite.tohoku.ac.jp/coda/tedit/help/index.html");
        charCode = prefs.get("CHAR_CODE", "UTF-8");
        preferredFontName = prefs.get("PREFERRED_FONT_NAME", "ＭＳ ゴシック");
        preferredFontSize = Integer.valueOf(prefs.get("PREFERRED_FONT_SIZE","14")) ;
        passcodeForFeedback = prefs.get("PASSCODE","") ;
        nicknameForFeedback = prefs.get("NICKNAME","UNNAMED") ;
        channelForFeedback = Integer.valueOf(prefs.get("CHANNEL","0")) ;
        
        turtleFieldPathName = prefs.get("TF_PATH_NAME","") ;
        String launchMode = prefs.get("TF_AUTO_LAUNCH","NO") ;
        turtleFieldAutoLaunch = launchMode.equals("YES");
        
    }

    public void setInterpreterMode(Boolean mode) {
        interpreterMode = mode ;
        compileButton.setEnabled(!mode) ;
        compileMenuItem.setEnabled(!mode) ;
    }
    
    public void setCompileCommand(String s) {
        compileCommand = s;
    }
    
    public void setExecCommand(String s) {
        execCommand = s;
    }
    
    public void setExecPath(String s) {
        execPath = s;
    }

    public void setHelpURI(String s) {
        helpURI = s;
    }

    public void setCharCode(String s) {
        charCode = s;
    }

    public void setPreferredFontName(String s) {
        preferredFontName = s ;
    }
    
    public void setPreferredFont(String name, int size) {
        if (! name.equals(preferredFontName) || size != preferredFontSize) {
            preferredFontName = name ;
            preferredFontSize = size ;
            setMonospacedFont() ;
        }
    }
    
    public void setTurtleField(String path, boolean mode) {
        turtleFieldPathName = path;
        turtleFieldAutoLaunch = mode;             
    }

       
    public String baseFileName(String s) {
        int pos = s.lastIndexOf('.');
        if (pos == -1) {
            return s;
        } else if (pos == 0) {
            return s;
        } else {
            return s.substring(0, pos);
        }
    }
    
     public void setNicknamePasscodeAndChannel(String nickname, String passcode, int channel) {
        nicknameForFeedback = nickname ;
        passcodeForFeedback = passcode ;
        channelForFeedback = channel ;
        prefs.put("NICKNAME", nickname);
        prefs.put("PASSCODE", passcode);
        prefs.put("CHANNEL",Integer.toString(channel)) ;
        try {
            prefs.flush();
        } catch (java.util.prefs.BackingStoreException e) {
            ;
        }
    }
    
    private int numberOfNewline(String str) {
        int i,j,cnt ;
        cnt=0 ;
        i=0 ;
        while ((j=str.indexOf('\n',i))>=0) {
            cnt++ ;
            i = j + 1 ;
        }
        return cnt ;
    }
    
    public void performFiltering(String regex, int type, boolean withLineNum, boolean dotAllFlag) {
        Pattern ptn ;
        if (dotAllFlag)  ptn = Pattern.compile(regex,Pattern.DOTALL|Pattern.MULTILINE) ;
        else ptn = Pattern.compile(regex,Pattern.MULTILINE) ;
        String target = editPane.getText() ; ;
        int start,next,ln ;
        Matcher m ;
        ((TEConsolePane)consolePane).clearAndSetText("") ;
        switch (type) {
            case 0:
                ln=1 ;
                start=0 ;
                m = ptn.matcher(target);
                while (m.find()) {
                    MatchResult res = m.toMatchResult();
                    String str = target.substring(res.start(), res.end());
                    if (withLineNum) {
                        ln += numberOfNewline(target.substring(start,res.start())) ;
                        ((TEConsolePane) consolePane).appendText(String.format("%1$04d",ln) + ":");
                    }
                    ((TEConsolePane) consolePane).appendText(str + "\n");
                    start=res.end() ;
                }
                break;
            case 1:
            case 2:
                ln = 1;
                start = 0;
                while ((next = target.indexOf('\n', start)) >= 0) {
                    // ((TEConsolePane) consolePane).appendText(start + " " + next + "\n");
                    String line = target.substring(start, next); 
                    m = ptn.matcher(line);
                    if ((type==1) ? m.matches() : m.find()) {
                        if (withLineNum) {
                            ((TEConsolePane) consolePane).appendText(String.format("%1$04d",ln) + ":");
                        }
                        ((TEConsolePane) consolePane).appendText(line + "\n");
                    }
                    start = next + 1;
                    ln++;
                }
                // remaining if any
                if (start<target.length()) {
                    next = target.length() ;
                    String line = target.substring(start, next); 
                    m = ptn.matcher(line);
                    if ((type==1) ? m.matches() : m.find()) {
                        if (withLineNum) {
                            ((TEConsolePane) consolePane).appendText(String.format("%1$04d",ln) + ":");
                        }
                        ((TEConsolePane) consolePane).appendText(line + "\n");
                    }
                }
                break;         
            default: break ;
        }
    }

    public void performReplacement(String regex, String replaceTo, boolean dotAllFlag){
        Pattern ptn ;
        if (dotAllFlag) ptn = Pattern.compile(regex,Pattern.DOTALL|Pattern.MULTILINE) ;
        else ptn = Pattern.compile(regex,Pattern.MULTILINE) ;
        String target = editPane.getText() ; ;
        Matcher m = ptn.matcher(target);
        String res = m.replaceAll(replaceTo) ;
        ((TEConsolePane)consolePane).clearAndSetText(res) ;  
    }
   
    public void performSort(int key, boolean ignoreCase, boolean reverseOrder, boolean numericSort) {
       String lines[] = editPane.getText().split("\\r?\\n") ;
       Arrays.sort(lines, new LineComparator(key,ignoreCase, reverseOrder,numericSort));
       StringBuilder txt = new StringBuilder() ;
       for (int i=0 ; i<lines.length ; i++) {
           txt.append(lines[i] + "\n") ;
       }
       ((TEConsolePane)consolePane).clearAndSetText(txt.toString()) ;
    }
    
    public void performUniq(boolean counting) {
        int counter=0 ;
        String lines[] = editPane.getText().split("\\r?\\n") ;
        Iterator iter = Arrays.asList(lines).iterator() ;
        StringBuilder txt = new StringBuilder() ;
        String previous=null ;
        while (iter.hasNext()) {
            String current = iter.next().toString() ; 
            if (previous == null || current.compareTo(previous)==0) {
                counter++ ;
            } else {
                if (counting) txt.append(counter + " ") ;
                txt.append(previous + "\n") ;
                counter=1 ;
            }
            previous = current ;
        }
        if (previous!=null) {
            if (counting) txt.append(counter + " ") ;
            txt.append(previous + "\n") ;
        }
        ((TEConsolePane)consolePane).clearAndSetText(txt.toString()) ;
    }
    
    public void performCounting(boolean linenum, boolean countword, boolean countchar, boolean appendtext) {
        String lines[] = editPane.getText().split("\\r?\\n") ;
        StringBuilder txt = new StringBuilder() ;
        int lineno=1 ;
        for (int i=0; i<lines.length ; i++) {
            if (linenum) txt.append(lineno + " ") ;
            if (countword) {
                int nword ;
                if (lines[i].length()==0) nword=0 ;
                else nword = lines[i].trim().split("\\s+").length ;
                txt.append(nword + " ") ;
            }
            if (countchar) {
                int nchar = lines[i].length() ;
                txt.append(nchar + " ") ;
            }
            if (appendtext) {
                txt.append(lines[i]) ;
            }
            txt.append("\n") ;
            lineno++ ;
        }
        ((TEConsolePane)consolePane).clearAndSetText(txt.toString()) ;
    }
    
    public void performTranslation(int trfunc) {
        String text = editPane.getText() ;
        String translatedText=text ;
        switch (trfunc) {
            case 1: // to lower case
                translatedText = text.toLowerCase() ;
                break ;
            case 2: // to upper case
                translatedText = text.toUpperCase() ;
                break ;
            case 3: // zenkaku to hankaku
                translatedText = zenkakuToHankaku(text) ;
                break ;
            case 4: // hankaku to zenkaku
                translatedText = hankakuToZenkaku(text) ;
                break ;
            default:
        }
        ((TEConsolePane)consolePane).clearAndSetText(translatedText) ;
    }
    
    public String zenkakuToHankaku(String str) {
        StringBuffer sb = new StringBuffer(str) ;
        for (int i=0; i<sb.length() ; i++) {
            char c = sb.charAt(i) ;
            if (c >='！' && c<='～') {
                sb.setCharAt(i, (char)(c - '！' + '!'));
            } else if (c=='　') {
                sb.setCharAt(i,' ');
            }
        }
        return sb.toString() ;
    }
    public String hankakuToZenkaku(String str) {
        StringBuffer sb = new StringBuffer(str) ;
        for (int i=0; i<sb.length() ; i++) {
            char c = sb.charAt(i) ;
            if (c >='!' && c<='~') {
                sb.setCharAt(i, (char)(c - '!' + '！'));
            } else if (c==' ') {
                sb.setCharAt(i,'　');
            }
        }
        return sb.toString() ;
    }
    
    private String convertQuotedSegment(String s) {
        char c0=0,c1 ;
        boolean quoted=false ;
        StringBuilder sb = new StringBuilder(s.length()) ;
        for (int i=0 ; i<s.length() ; i++) {
            c1 = s.charAt(i) ;
            if (c0!='\\' && c1=='\'') {
                if (!quoted) quoted=true ;
                else quoted=false ;
            } else if (c0=='\\' && c1=='\'') {
                sb.deleteCharAt(sb.length()-1);
                sb.append(c1);
            } else {
                if (c1 == ' ') {
                    if (quoted) {
                        sb.append("%s");
                    } else {
                        sb.append(" ");
                    }
                } else {
                    sb.append(c1);
                }
            }
            c0 = c1 ;
        }
        return sb.toString() ;
    }
    
    // In Window path names are returned in a form like "C:\Users\foo..."
    // so conversion from "\" to "\\" is necessary before passing it to
    // ProcessBuilder
    public List parseCommandLine(String s) {
        List commandList = new ArrayList();
        String s2 = convertQuotedSegment(s) ;
        StringTokenizer tokenizer = new StringTokenizer(s2, " ");
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            if (workingFile != null) {
                token = token.replaceAll("%f", workingFile.getName());
                token = token.replaceAll("%d", workingFile.getParent().replaceAll("\\\\", "\\\\\\\\"));
                token = token.replaceAll("%b", baseFileName(workingFile.getName()));
            }
            token = token.replaceAll("%c", new File(".").getAbsoluteFile().getParent().replaceAll("\\\\","\\\\\\\\")) ;
            token = token.replaceAll("%h", System.getProperty("user.home").replaceAll("\\\\","\\\\\\\\")) ;
            token = token.replaceAll("%s", " ") ;
            commandList.add(token);
        }
        return commandList;
    }

    public String checkCompileError() {
        int rc = -1;
        compileWorkingFile();
        do {
            if (runningProcess!=null) {
                try {
                    rc = runningProcess.exitValue();
                } catch (IllegalThreadStateException e) {
                    rc = -1;
                }
            } else break ;
            try {
                Thread.sleep(200); /* wait 200ms */
            } catch (InterruptedException ex) {
                ;
            }
        } while(rc<0) ;
        String result = consolePane.getText() ;
        // System.out.println(result) ;
        return result ;
    }
    
    public void compileWorkingFile() {
        if (confirmToSaveDialogue() == 0) {
            return;
        }
        if (!workingFileIsSet) {
            ((TEConsolePane) consolePane).clearAndSetText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("YOU NEED TO WRITE SOURCE CODE AND SAVE IT BEF0RE COMPILING"));
            return;
        }

        startCompiling = true ;
        compileWasDoneNormally = false ;
        int rc = execCommandInConsolePane(
                compileCommand, 
                java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("COMPILING"),
                false );

    }

    public void execCompiledCommand() {
        int rc = -1 ;
        
        if (interpreterMode) {
            if (confirmToSaveDialogue() == 0) {
                return;
            }
            if (!workingFileIsSet) {
                ((TEConsolePane) consolePane).clearAndSetText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("YOU NEED TO WRITE SOURCE CODE AND SAVE IT BEFORE EXECUTION"));
                return;
            }
        } else {
            if (startCompiling) {
                if (runningProcess == null) { // seems compile was terminated
                    startCompiling = false;
                } else {
                    try {
                        rc = runningProcess.exitValue();
                    } catch (IllegalThreadStateException e) {
                        rc = -1;
                    }
                }
                if (rc == 0) {
                    compileWasDoneNormally = true;
                }
            } else {
                if (((TEEditorPane) editPane).isContentModified()) { // contents in buffer not compiled yet
                    compileWasDoneNormally = false;
                }
            }

            if (!compileWasDoneNormally) {
                ((TEConsolePane) consolePane).clearAndSetText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("YOU NEED TO COMPILE SOURCE PROGRAM FIRST"));
                return;
            }
            startCompiling = false;
        }

        if (turtleFieldAutoLaunch) {
            tfLauncher.setCommandPath(turtleFieldPathName);
            tfLauncher.launch();
        }
        
        execCommandInConsolePane(
                execCommand, 
                java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("RUNNING PROGRAM"),
                false);
    }

    public void execCommandLine(String commandLine, boolean plungStdin) {
        ((TEConsolePane) consolePane).clearAndSetText("");
        execCommandInConsolePane(commandLine, "", true) ;
        if (plungStdin) {
           ((TEConsolePane) consolePane).writeStringToRunningProcess(
                   ((TEEditorPane) editPane).getText(), 
                   true /* close after writing */
                   ) ; 
        }
    } 
     
    public int execCommandInConsolePane(String commandLine, String startingMessage, boolean silentMode) {

        int rc = 0;

        String delimiter;
        String osname = System.getProperty("os.name");
        if (osname.indexOf("Windows") >= 0) {
            delimiter = ";";
        } else {
            delimiter = ":";
        }

        killRunningProcess(silentMode);
        try {
            Thread.sleep(2); /* wait 1ms */
        } catch (InterruptedException ex) {
            return -1;
        }

        List commandList = parseCommandLine(commandLine);

        ProcessBuilder pb = new ProcessBuilder(commandList);

        Map<String, String> env = pb.environment();
        String execPaths = env.get("PATH") + delimiter + execPath;
        env.remove("PATH");
        env.put("PATH", execPaths);

        pb.redirectErrorStream(true);
        if (workingFile!=null) {
            pb.directory(workingFile.getParentFile());
        } else {
            pb.directory(new File(System.getProperty("user.home")));
        }
        try {
            if (!silentMode) {
                ((TEConsolePane) consolePane).clearAndSetText(startingMessage + "\n$ ");
                Iterator iter = commandList.iterator();
                while (iter.hasNext()) {
                    String str = (String) iter.next();
                    ((TEConsolePane) consolePane).appendText(str + " ");
                }
                ((TEConsolePane) consolePane).appendText("\n");
            }
            runningProcess = pb.start();
            ((TEConsolePane) consolePane).handleProcess(runningProcess, charCode, silentMode);

        } catch (IOException ex) {
            ((TEConsolePane) consolePane).clearAndSetText("Command execution error: " + commandLine +"\n");
            String errmsg = headingLineOfStackTrace(ex) ;
            String msgbody = errmsg.replaceAll("^.*?:\\s*","") ;
            ((TEConsolePane) consolePane).appendText(msgbody+"\n") ;
            rc = -1;
        }
        return rc;
    }

    public String headingLineOfStackTrace(Exception ex) {
        StringWriter sw = new StringWriter() ;
            PrintWriter pw = new PrintWriter(sw) ;
            ex.printStackTrace(pw);
            pw.flush();
            Scanner sn = new Scanner(sw.toString()) ;
            if (sn.hasNextLine()) return sn.nextLine() ;
            else return "" ;
    }
    
    public void killRunningProcess(boolean silentMode) {
        if (runningProcess != null) {
            runningProcess.destroy();
            try {
                sleep(0,1000) ;
                ((TEConsolePane) consolePane).terminateRunningThread(); 
            } catch (InterruptedException ex) {
                ;
            }
            if (!silentMode) {
                 ((TEConsolePane) consolePane).appendText(java.util.ResourceBundle.getBundle("turtleedit/TEBandle").getString("INTERRUPTED")) ;
            }   
        }
        runningProcess = null;
    }

    public String checkAvailableMonospacedFont() {
         List<String> monospacedFonts = new ArrayList();
        String fonts[] = GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
        // picking up better fonts for programming...
        for (int i = 0; i < fonts.length; i++) {   
            // ((TEConsolePane)consolePane).appendText(fonts[i]+"\n");
            if (fonts[i].equalsIgnoreCase(preferredFontName)
                    || fonts[i].equalsIgnoreCase("MS Gothic")
                    || fonts[i].equalsIgnoreCase("ＭＳ ゴシック")
                    || fonts[i].equalsIgnoreCase("さざなみゴシック")
                    || fonts[i].equalsIgnoreCase("Monospaced")
                    || fonts[i].equalsIgnoreCase("Liberation Mono")
                    || fonts[i].equalsIgnoreCase("Lucida Console")
                    || fonts[i].equalsIgnoreCase("Monaco")
                    || fonts[i].equalsIgnoreCase("Terminal")
                    || fonts[i].equalsIgnoreCase("Courier")
                    || fonts[i].equalsIgnoreCase("Bitstream Vera Sans Mono")
                    || fonts[i].equalsIgnoreCase("OCRB")) {
                monospacedFonts.add(fonts[i]);
            }
        }
        
        // on Windows system, FontMetrics of non-Japanese fonts don't give correct width
        // for multi-byte characters... MS Gothic seems to ba a safe choice so that
        // has to be placed in the first.
        if (monospacedFonts.contains(preferredFontName)) {
            return preferredFontName ;
        } else if (monospacedFonts.contains("MS Gothic")) {
            return "MS Gothic" ;
        } else if (monospacedFonts.contains("ＭＳ ゴシック")) {
            return "ＭＳ ゴシック";
        } else if (monospacedFonts.contains("さざなみゴシック")) {
            return "さざなみゴシック";
        // } else if (monospacedFonts.contains("Liberation Mono")) {
        //     return "Liberation Mono";
        } else if (monospacedFonts.contains("Lucida Console")) {
            return "Lucida Console";
        } else if (monospacedFonts.contains("Bitstream Vera Sans Mono")) {
            return "Bitstream Vera Sans Mono";
        } else if (monospacedFonts.contains("Monaco")) {
            return "Monaco";
        } else if (monospacedFonts.contains("Terminal")) {
            return "Terminal";
        } else if (monospacedFonts.contains("Monospaced")) {
            return "Monospaced";
        } else if (monospacedFonts.contains("OCRB")) {
            return "OCRB";
        } else if (monospacedFonts.contains("Courier")) {
            return "Courier";
        } else {
            return Font.MONOSPACED;
        }
    }

    public void setMonospacedFont() {
        String fontname = checkAvailableMonospacedFont();
        ((TEEditorPane) editPane).setTextFont(new Font(fontname, Font.PLAIN, preferredFontSize));
        ((TEEditorPane) editPane).setLineNumFont(new Font(fontname, Font.PLAIN, preferredFontSize));
        consolePane.setFont(new Font(fontname, Font.PLAIN, 12));
    }
    
    public void distillClipboardContent() {
        Toolkit kit = Toolkit.getDefaultToolkit();
        Clipboard cb = kit.getSystemClipboard();

        try {
            String str = (String) cb.getData(DataFlavor.stringFlavor);
            StringSelection ss = new StringSelection(str) ;
            cb.setContents(ss, ss);
        } catch (UnsupportedFlavorException e) {
            return ;
        } catch (IOException e) {
            return ;
        }
    }

    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JMenuItem aboutMenuItem;
    private javax.swing.JButton breakButton;
    private javax.swing.JMenuItem breakMenuItem;
    private javax.swing.JButton compileButton;
    private javax.swing.JMenuItem compileMenuItem;
    private javax.swing.JTextPane consolePane;
    private javax.swing.JMenuItem copyMenuItem;
    private javax.swing.JMenuItem countMenuItem;
    private javax.swing.JMenuItem cutMenuItem;
    private javax.swing.JEditorPane editPane;
    private javax.swing.JMenuItem execCommandMenuItem;
    private javax.swing.JMenuItem exitMenuItem;
    private javax.swing.JMenuItem feedbackMenuItem;
    private javax.swing.Box.Filler filler1;
    private javax.swing.JMenu filterMenu;
    private javax.swing.JMenuItem findMenuItem;
    private javax.swing.JMenuItem findNextMenuItem;
    private javax.swing.JMenuItem findPreviousMenuItem;
    private javax.swing.JMenu jMenu1;
    private javax.swing.JMenu jMenu2;
    private javax.swing.JMenu jMenu3;
    private javax.swing.JMenu jMenu4;
    private javax.swing.JMenu jMenu5;
    private javax.swing.JMenuBar jMenuBar1;
    private javax.swing.JMenuItem jMenuItem1;
    private javax.swing.JMenuItem jMenuItem2;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JScrollPane jScrollPane3;
    private javax.swing.JPopupMenu.Separator jSeparator1;
    private javax.swing.JPopupMenu.Separator jSeparator2;
    private javax.swing.JPopupMenu.Separator jSeparator4;
    private javax.swing.JPopupMenu.Separator jSeparator5;
    private javax.swing.JPopupMenu.Separator jSeparator6;
    private javax.swing.JMenuItem newMenuItem;
    private javax.swing.JMenuItem openMenuItem;
    private javax.swing.JMenuItem pasteMenuItem;
    private javax.swing.JButton pwdButton;
    private javax.swing.JMenuItem regExFilterMenuItem;
    private javax.swing.JMenuItem regExReplaceMenuItem;
    private javax.swing.JButton runButton;
    private javax.swing.JMenuItem runMenuItem;
    private javax.swing.JMenuItem saveAsMenuItem;
    private javax.swing.JMenuItem saveMenuItem;
    private javax.swing.JMenuItem saveOutputsMenuItem;
    private javax.swing.JMenuItem settingsMenuItem;
    private javax.swing.JMenuItem sortMenuItem;
    private javax.swing.JMenuItem swapBuffersMenuItem;
    private javax.swing.JSplitPane terminalSplitPane;
    private javax.swing.JMenuItem translationMenuItem;
    private javax.swing.JMenuItem undoMenuItem;
    private javax.swing.JMenuItem uniqMenuItem;
    private javax.swing.JMenuItem webHelpMenuItem;
    // End of variables declaration//GEN-END:variables

    class LineComparator implements Comparator<String> {
        private int key ;
        private boolean ignoreCase ;
        private boolean reverseOrder ;
        private boolean numericSort ;
        
        LineComparator(int k, boolean caseinsensitive, boolean reverse, boolean numeric) {
            key = k ;
            ignoreCase=caseinsensitive ;
            reverseOrder=reverse ;
            numericSort=numeric ;
        }
        
        @Override
        public int compare(String o1, String o2) {
            int res=0 ;
            String key1,key2 ;
            if (key == 0) {
                key1 = o1;
                key2 = o2;
            } else {
                String s1[] = o1.trim().split("\\s+",5) ;
                String s2[] = o2.trim().split("\\s+",5) ;
                if (s1.length >= key) {
                    key1 = s1[key - 1];
                } else {
                    key1 = "";
                }
                if (s2.length >= key) {
                    key2 = s2[key - 1];
                } else {
                    key2 = "";
                }
            }
            
            if (numericSort) {
                double val1,val2 ;
                try {
                    val1 = Double.parseDouble(key1);
                    val2 = Double.parseDouble(key2);
                } catch (NumberFormatException e) {
                    val1 = val2 = 0 ;
                }
                if (val1 > val2) res=1 ;
                else if (val1<val2) res=-1 ;
                else res=0 ;
                
            } else {
                if (ignoreCase)
                    res = key1.compareToIgnoreCase(key2);
                else
                    res = key1.compareTo(key2);
            }        
            if (reverseOrder) res = -res ;
            return res ;
        }
    }
}
